MODULE module_diagvar_cordex
! Module with the specific subroutines and functions to compute the CORDEX required variables
! L. Fita, CIMA. December 2017
!
  USE module_model_constants

!! These are the definition of parameters used in the calculations
!   They might be placed on the general WRF's share/module_model_constants.F ... ?
  ! Current version of the module
  CHARACTER(LEN=50)                                      :: cdxwrfversion = '1.3'

  INTEGER, PARAMETER                                     :: r_k = KIND(1.d0)

  REAL(r_k), PARAMETER                                   :: ZEPSEC=1.0D-12
! Low limit pressure for medium clouds [Pa]
  REAL(r_k), PARAMETER                                   :: prmhc = 44000.d0
! Low limit pressure for High clouds [Pa]
  REAL(r_k), PARAMETER                                   :: prmlc = 68000.d0
! Real values for cloud intervals
  REAL, PARAMETER                                        :: prmhcR = prmhc*1.
  REAL, PARAMETER                                        :: prmlcR = prmlc*1.
! August-Roche-Magnus approximation
  REAL, PARAMETER                                        :: ARM1 = 6.1094
  REAL, PARAMETER                                        :: ARM2 = 17.625
  REAL, PARAMETER                                        :: ARM3 = 243.04

! Ratio between molecular weights of water and dry air
  REAL, PARAMETER                                        :: mol_watdry = 0.622
! Standard atmospheric lapse-rate [Km-1]
  REAL, PARAMETER                                        :: gammav=6.5e-3

  REAL(r_k), PARAMETER                                   :: zeroRK=0.d0
  REAL(r_K), PARAMETER                                   :: halfRK=0.5d0
  REAL(r_K), PARAMETER                                   :: oneRK=1.d0
  REAL(r_k), PARAMETER                                   :: twoRK=2.d0
! FillValue
  REAL, PARAMETER                                        :: fillValueR=1.e20

  CHARACTER(LEN=50)                                      :: sfname, IS, RS
  CHARACTER (LEN=256)                                    :: message
  CHARACTER (LEN=1000)                                   :: Lmessage

! Zero
!  REAL, PARAMETER                                        :: zero=10.e-6 

! null value
  REAL(r_k)                                              :: nullv = 1.e-7
  !REAL(r_k)                                              :: nullv = 1.d-15

  CONTAINS

!!!!!!! diagnosted variables
! var_cape_afwa: CAPE following AFWA methodology
! var_cllmh: low, medium, high-cloud [0,100]
! var_clt: total cloudiness [0,100]
! var_fog_K84: Computation of fog and visibility following Kunkel, (1984)
! var_fog_RUC: Computation of fog and visibility following RUC method Smirnova, (2000)
! var_fog_FRAML50: fog and visibility following Gultepe and Milbrandt, (2010)
! var_hur: relative humidity using August-Roche-Magnus approximation [1]
! var_hurs: relative humidity at 2m using August-Roche-Magnus approximation [1]
! var_hus: specific humidity [1]
! var_huss: 2m specific humidity [1]
! var_potevap_bulk: potential evapotranspiration following simple bulk formulation
! var_potevap_Milly92: potential bulk evapotranspiration with Milly 1992 correction
! var_potevap_bulkgen: generic potential evapotranspiration following simple bulk formulation
! var_potevap_Milly92gen: genreic potential bulk evapotranspiration with Milly 1992 correction
! var_press: air pressure [Pa]
! var_psl_ecmwf: sea level pressure using ECMWF method
! var_psl_ptarget: mean sea level pressure using a target pressure
! var_psl_shuell: mean sea level pressure using the hydrostatic equation with the Shuell correction
! var_massvertint: Subroutine to vertically integrate a 1D variable in eta vertical coordinates
! var_vertint: Subroutine to vertically integrate a 1D variable in any vertical coordinates
! var_ta: air temperature [K]
! var_tds: dew point temperature at 2m using August-Roche-Magnus approximation [K]
! var_uava: earth-rotated wind components [ms-1]
! var_uasvas: 10m earth-rotated wind components [ms-1]
! var_zg: geopotential height [m]
! var_zmla_generic: pbl-height following a generic method [m]
! var_zwind: extrapolate the wind at a given height following the 'power law' methodology
! var_zwind_log: extrapolate the wind at a given height following the 'logarithmic law' methodology
! var_zwind_MOtheor: wind extrapolation following Moin-Obukhov theory
! water_budget: Subroutine to compute accumulated water budget
! gustwind_Brasseur01: Wind gust following Brasseur (2001), MWR. 
! gustwind_afwa: Wind gust following heavy precip methodology

!!!
! Variables
!!!
! Cdrag_0: first order generic approximation of the drag coefficient
! Theta: potential temperature
! Thetae: equivalent potential temperature
! The2T: temperature at any pressure level along a saturation adiabat
! VirtualTemperature: virtual temperature
! SaturationMixingRatio: saturation mixing ratio
! TLCL: Temperature of a parcel of air would have dry adiabatically lifted to it's lcl
! VirtPotTemp: Function to compute virtual potential temperature

!!!
! Operations
!!!
! interval_agg: Subroutine to compute aggregation by intervals along the full axis of of mass-values
! moist_group: Subroutine to group 1D values from each water-species to a single variable
! moist_group2D: Subroutine to group 2D values from each water-species to a single variable
! moist_redistribute: Subroutine to re-distribute a multi water-species 1D value to each specie
! moist_redistribute2D: Subroutine to re-distribute a multi water-species 2D value to each specie
! partial_agg: Subroutine to compute a partial aggregation of mass-values between an interval
! NumIntegration: Subroutine to compute numerical integrations according to the trapezoidal methodology

! L. Fita, CIMA. Jan 2018
! ---- BEGIN modified from module_diag_afwa.F ---- !

  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~
  !~ Name:
  !~    Theta
  !~
  !~ Description:
  !~    This function calculates potential temperature as defined by
  !~    Poisson's equation, given temperature and pressure ( hPa ).
  !~
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION Theta ( t, p )
  IMPLICIT NONE

     !~ Variable declaration
     !  --------------------
     REAL, INTENT ( IN ) :: t
     REAL, INTENT ( IN ) :: p
     REAL                :: theta

     ! Using WRF values
     !REAL :: Rd ! Dry gas constant
     !REAL :: Cp ! Specific heat of dry air at constant pressure
     !REAL :: p00 ! Standard pressure ( 1000 hPa )
     REAL                                                :: Rd, p00
  
     !Rd =  287.04
     !Cp = 1004.67
     !p00 = 1000.00

     Rd = r_d
     p00 = p1000mb/100.

     !~ Poisson's equation
     !  ------------------
     theta = t * ( (p00/p)**(Rd/Cp) )
  
  END FUNCTION Theta



  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~
  !~ Name:
  !~    Thetae
  !~
  !~ Description:
  !~    This function returns equivalent potential temperature using the 
  !~    method described in Bolton 1980, Monthly Weather Review, equation 43.
  !~
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION Thetae ( tK, p, rh, mixr )
  IMPLICIT NONE

     !~ Variable Declarations
     !  ---------------------
     REAL :: tK        ! Temperature ( K )
     REAL :: p         ! Pressure ( hPa )
     REAL :: rh        ! Relative humidity
     REAL :: mixr      ! Mixing Ratio ( kg kg^-1)
     REAL :: te        ! Equivalent temperature ( K )
     REAL :: thetae    ! Equivalent potential temperature
  
     ! Using WRF values
     !REAL, PARAMETER :: R  = 287.04         ! Universal gas constant (J/deg kg)
     !REAL, PARAMETER :: P0 = 1000.0         ! Standard pressure at surface (hPa)
     REAL                                                :: R, p00, Lv
     !REAL, PARAMETER :: lv = 2.54*(10**6)   ! Latent heat of vaporization
                                            ! (J kg^-1)
     !REAL, PARAMETER :: cp = 1004.67        ! Specific heat of dry air constant
                                            ! at pressure (J/deg kg)
     REAL :: tlc                            ! LCL temperature
  
     R = r_d
     p00 = p1000mb/100.
     lv = XLV

     !~ Calculate the temperature of the LCL
     !  ------------------------------------
     tlc = TLCL ( tK, rh )
  
     !~ Calculate theta-e
     !  -----------------
     thetae = (tK * (p00/p)**( (R/Cp)*(1.- ( (.28E-3)*mixr*1000.) ) ) )* &
                 exp( (((3.376/tlc)-.00254))*&
                    (mixr*1000.*(1.+(.81E-3)*mixr*1000.)) )
  
  END FUNCTION Thetae



  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~
  !~ Name:
  !~    The2T.f90
  !~
  !~ Description:
  !~    This function returns the temperature at any pressure level along a
  !~    saturation adiabat by iteratively solving for it from the parcel
  !~    thetae.
  !~
  !~ Dependencies:
  !~    function thetae.f90
  !~
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION The2T ( thetaeK, pres, flag ) result ( tparcel )
  IMPLICIT NONE
  
     !~ Variable Declaration
     !  --------------------
     REAL,    INTENT     ( IN ) :: thetaeK
     REAL,    INTENT     ( IN ) :: pres
     LOGICAL, INTENT ( INOUT )  :: flag
     REAL                       :: tparcel
  
     REAL :: thetaK
     REAL :: tovtheta
     REAL :: tcheck
     REAL :: svpr, svpr2
     REAL :: smixr, smixr2
     REAL :: thetae_check, thetae_check2
     REAL :: tguess_2, correction
  
     LOGICAL :: found
     INTEGER :: iter
  
     ! Using WRF values
     !REAL :: R     ! Dry gas constant
     !REAL :: Cp    ! Specific heat for dry air
     !REAL :: kappa ! Rd / Cp
     !REAL :: Lv    ! Latent heat of vaporization at 0 deg. C
     REAL                                                :: R, kappa, Lv

     R = r_d
     Lv = XLV
     !R     = 287.04
     !Cp    = 1004.67
     Kappa = R/Cp
     !Lv    = 2.500E+6

     !~ Make initial guess for temperature of the parcel
     !  ------------------------------------------------
     tovtheta = (pres/100000.0)**(r/cp)
     tparcel  = thetaeK/exp(lv*.012/(cp*295.))*tovtheta

     iter = 1
     found = .false.
     flag = .false.

     DO
        IF ( iter > 105 ) EXIT

        tguess_2 = tparcel + REAL ( 1 )

        svpr   = 6.122 * exp ( (17.67*(tparcel-273.15)) / (tparcel-29.66) )
        smixr  = ( 0.622*svpr ) / ( (pres/100.0)-svpr )
        svpr2  = 6.122 * exp ( (17.67*(tguess_2-273.15)) / (tguess_2-29.66) )
        smixr2 = ( 0.622*svpr2 ) / ( (pres/100.0)-svpr2 )

        !  ------------------------------------------------------------------ ~!
        !~ When this function was orinially written, the final parcel         ~!
        !~ temperature check was based off of the parcel temperature and      ~!
        !~ not the theta-e it produced.  As there are multiple temperature-   ~!
        !~ mixing ratio combinations that can produce a single theta-e value, ~!
        !~ we change the check to be based off of the resultant theta-e       ~!
        !~ value.  This seems to be the most accurate way of backing out      ~!
        !~ temperature from theta-e.                                          ~!
        !~                                                                    ~!
        !~ Rentschler, April 2010                                             ~!
        !  ------------------------------------------------------------------  !

        !~ Old way...
        !thetaK = thetaeK / EXP (lv * smixr  /(cp*tparcel) )
        !tcheck = thetaK * tovtheta

        !~ New way
        thetae_check  = Thetae ( tparcel,  pres/100., 100., smixr  )
        thetae_check2 = Thetae ( tguess_2, pres/100., 100., smixr2 )

        !~ Whew doggies - that there is some accuracy...
        !IF ( ABS (tparcel-tcheck) < .05) THEN
        IF ( ABS (thetaeK-thetae_check) < .001) THEN
           found = .true.
           flag  = .true.
           EXIT
        END IF

        !~ Old
        !tparcel = tparcel + (tcheck - tparcel)*.3

        !~ New
        correction = ( thetaeK-thetae_check ) / ( thetae_check2-thetae_check )
        tparcel = tparcel + correction

        iter = iter + 1
     END DO

     !IF ( .not. found ) THEN
     !   print*, "Warning! Thetae to temperature calculation did not converge!"
     !   print*, "Thetae ", thetaeK, "Pressure ", pres
     !END IF

  END FUNCTION The2T



  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~
  !~ Name:
  !~    VirtualTemperature
  !~
  !~ Description:
  !~    This function returns virtual temperature given temperature ( K )
  !~    and mixing ratio.
  !~
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION VirtualTemperature ( tK, w ) result ( Tv )
  IMPLICIT NONE

     !~ Variable declaration
     real, intent ( in ) :: tK !~ Temperature
     real, intent ( in ) :: w  !~ Mixing ratio ( kg kg^-1 )
     real                :: Tv !~ Virtual temperature

     Tv = tK * ( 1.0 + (w/0.622) ) / ( 1.0 + w )

  END FUNCTION VirtualTemperature




  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~
  !~ Name:
  !~    SaturationMixingRatio
  !~
  !~ Description:
  !~    This function calculates saturation mixing ratio given the
  !~    temperature ( K ) and the ambient pressure ( Pa ).  Uses 
  !~    approximation of saturation vapor pressure.
  !~
  !~ References:
  !~    Bolton (1980), Monthly Weather Review, pg. 1047, Eq. 10
  !~
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION SaturationMixingRatio ( tK, p ) result ( ws )

    IMPLICIT NONE

    REAL, INTENT ( IN ) :: tK
    REAL, INTENT ( IN ) :: p
    REAL                :: ws

    REAL :: es

    es = 6.122 * exp ( (17.67*(tK-273.15))/ (tK-29.66) )
    ws = ( 0.622*es ) / ( (p/100.0)-es )

  END FUNCTION SaturationMixingRatio

  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~                                                                     
  !~ Name:                                                                
  !~    tlcl                                                               
  !~                                                                        
  !~ Description:                                                            
  !~    This function calculates the temperature of a parcel of air would have
  !~    if lifed dry adiabatically to it's lifting condensation level (lcl).  
  !~                                                                          
  !~ References:                                                              
  !~    Bolton (1980), Monthly Weather Review, pg. 1048, Eq. 22
  !~                                                                          
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION TLCL ( tk, rh )
    
    IMPLICIT NONE
 
    REAL, INTENT ( IN ) :: tK   !~ Temperature ( K )
    REAL, INTENT ( IN ) :: rh   !~ Relative Humidity ( % )
    REAL                :: tlcl
    
    REAL :: denom, term1, term2

    term1 = 1.0 / ( tK - 55.0 )
    IF ( rh > REAL (0) ) THEN
      term2 = ( LOG (rh/100.0)  / 2840.0 )
    ELSE
      term2 = ( LOG (0.001/1.0) / 2840.0 )
    END IF
    denom = term1 - term2
    tlcl = ( 1.0 / denom ) + REAL ( 55 ) 

  END FUNCTION TLCL

! Changing to a subroutine
!  FUNCTION var_cape_afwa(nz, tk, rhv, p, hgt, sfc, cape, cin, zlfc, plfc, lidx, parcel) RESULT (ostat)
  SUBROUTINE var_cape_afwa(nz, tk, rhv, p, hgt, sfc, cape, cin, zlfc, plfc, lidx, parcel)
! Function to compute cape on a 1D column following implementation in phys/module_diag_afwa.F

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: nz, sfc
    REAL, DIMENSION(nz), INTENT(in)                      :: tk, rhv, p, hgt
    REAL, INTENT(out)                                    :: cape, cin, zlfc, plfc, lidx
    INTEGER                                              :: ostat
    INTEGER, INTENT(in)                                  :: parcel
  
    ! Local
    !~ Derived profile variables
    !  -------------------------
    REAL, DIMENSION(nz)                                  :: rh, ws, w, dTvK, buoy
    REAL                                                 :: tlclK, plcl, nbuoy, pbuoy
  
    !~ Source parcel information
    !  -------------------------
    REAL                                                 :: srctK, srcrh, srcws, srcw, srcp,          &
      srctheta, srcthetaeK
    INTEGER                                              :: srclev
    REAL                                                 :: spdiff
   
    !~ Parcel variables
    !  ----------------
    REAL                                                 :: ptK, ptvK, tvK, pw
  
    !~ Other utility variables
    !  -----------------------
    INTEGER                                              :: i, j, k
    INTEGER                                              :: lfclev
    INTEGER                                              :: prcl
    INTEGER                                              :: mlev
    INTEGER                                              :: lyrcnt
    LOGICAL                                              :: flag
    LOGICAL                                              :: wflag
    REAL                                                 :: freeze
    REAL                                                 :: pdiff
    REAL                                                 :: pm, pu, pd
    REAL                                                 :: lidxu
    REAL                                                 :: lidxd
  
    REAL, PARAMETER                                      :: Rd = r_d
    REAL, PARAMETER                                      :: RUNDEF = -9.999E30

!!!!!!! Variables
! nz: Number of vertical levels
! sfc: Surface level in the profile
! tk: Temperature profile [K]
! rhv: Relative Humidity profile [1]
! rh: Relative Humidity profile [%]
! p: Pressure profile [Pa]
! hgt: Height profile [gpm]
! cape: CAPE [Jkg-1]
! cin: CIN [Jkg-1]
! zlfc: LFC Height [gpm]
! plfc: LFC Pressure [Pa]
! lidx: Lifted index
!   FROM: https://en.wikipedia.org/wiki/Lifted_index
!     lidx >= 6: Very Stable Conditions
!     6 > lidx > 1: Stable Conditions, Thunderstorms Not Likely
!     0 > lidx > -2: Slightly Unstable, Thunderstorms Possible, With Lifting Mechanism (i.e., cold front, daytime heating, ...)
!     -2 > lidx > -6: Unstable, Thunderstorms Likely, Some Severe With Lifting Mechanism
!     -6 > lidx: Very Unstable, Severe Thunderstorms Likely With Lifting Mechanism
! ostat: Function return status (Nonzero is bad)
! parcel:
!   Most Unstable = 1 (default)
!   Mean layer = 2
!   Surface based = 3
!~ Derived profile variables
!  -------------------------
! ws: Saturation mixing ratio
! w: Mixing ratio
! dTvK: Parcel / ambient Tv difference
! buoy: Buoyancy
! tlclK: LCL temperature [K]
! plcl: LCL pressure [Pa]
! nbuoy: Negative buoyancy
! pbuoy: Positive buoyancy
  
!~ Source parcel information
!  -------------------------
! srctK: Source parcel temperature [K]
! srcrh: Source parcel rh [%]
! srcws: Source parcel sat. mixing ratio
! srcw: Source parcel mixing ratio
! srcp: Source parcel pressure [Pa]
! srctheta: Source parcel theta [K]
! srcthetaeK: Source parcel theta-e [K]
! srclev: Level of the source parcel
! spdiff: Pressure difference
   
!~ Parcel variables
!  ----------------
! ptK: Parcel temperature [K]
! ptvK: Parcel virtual temperature [K]
! tvK: Ambient virtual temperature [K]
! pw: Parcel mixing ratio
  
!~ Other utility variables
!  -----------------------
! lfclev: Level of LFC
! prcl: Internal parcel type indicator
! mlev: Level for ML calculation
! lyrcnt: Number of layers in mean layer
! flag: Dummy flag
! wflag: Saturation flag
! freeze: Water loading multiplier
! pdiff: Pressure difference between levs 
! pm, pu, pd: Middle, upper, lower pressures
! lidxu: Lifted index at upper level
! lidxd: Lifted index at lower level

    sfname = 'var_cape_afwa'  

    !~ Initialize variables
    !  --------------------
    rh = rhv*100.
    ostat = 0
    CAPE = REAL(0)
    CIN = REAL(0)
    ZLFC = RUNDEF
    PLFC = RUNDEF
  
    !~ Look for submitted parcel definition
    !~ 1 = Most unstable
    !~ 2 = Mean layer
    !~ 3 = Surface based
    !  -------------------------------------
    IF ( parcel > 3 .or. parcel < 1 ) THEN
       prcl = 1
    ELSE
       prcl =  parcel
    END IF
  
    !~ Initalize our parcel to be (sort of) surface based.  Because of
    !~ issues we've been observing in the WRF model, specifically with
    !~ excessive surface moisture values at the surface, using a true
    !~ surface based parcel is resulting a more unstable environment
    !~ than is actually occuring.  To address this, our surface parcel
    !~ is now going to be defined as the parcel between 25-50 hPa
    !~ above the surface. UPDATE - now that this routine is in WRF,
    !~ going to trust surface info. GAC 20140415
    !  ----------------------------------------------------------------
  
    !~ Compute mixing ratio values for the layer
    !  -----------------------------------------
    DO k = sfc, nz
      ws  ( k )   = SaturationMixingRatio ( tK(k), p(k) )
      w   ( k )   = ( rh(k)/100.0 ) * ws ( k )
    END DO
  
    srclev      = sfc
    srctK       = tK    ( sfc )
    srcrh       = rh    ( sfc )
    srcp        = p     ( sfc )
    srcws       = ws    ( sfc )
    srcw        = w     ( sfc )
    srctheta    = Theta ( tK(sfc), p(sfc)/100.0 )
   
      !~ Compute the profile mixing ratio.  If the parcel is the MU parcel,
      !~ define our parcel to be the most unstable parcel within the lowest
      !~ 180 mb.
      !  -------------------------------------------------------------------
      mlev = sfc + 1
      DO k = sfc + 1, nz
   
         !~ Identify the last layer within 100 hPa of the surface
         !  -----------------------------------------------------
         pdiff = ( p (sfc) - p (k) ) / REAL ( 100 )
         IF ( pdiff <= REAL (100) ) mlev = k

         !~ If we've made it past the lowest 180 hPa, exit the loop
         !  -------------------------------------------------------
         IF ( pdiff >= REAL (180) ) EXIT

         IF ( prcl == 1 ) THEN
            !IF ( (p(k) > 70000.0) .and. (w(k) > srcw) ) THEN
            IF ( (w(k) > srcw) ) THEN
               srctheta = Theta ( tK(k), p(k)/100.0 )
               srcw = w ( k )
               srclev  = k
               srctK   = tK ( k )
               srcrh   = rh ( k )
               srcp    = p  ( k )
            END IF
         END IF
   
      END DO
   
      !~ If we want the mean layer parcel, compute the mean values in the
      !~ lowest 100 hPa.
      !  ----------------------------------------------------------------
      lyrcnt =  mlev - sfc + 1
      IF ( prcl == 2 ) THEN
   
         srclev   = sfc
         srctK    = SUM ( tK (sfc:mlev) ) / REAL ( lyrcnt )
         srcw     = SUM ( w  (sfc:mlev) ) / REAL ( lyrcnt )
         srcrh    = SUM ( rh (sfc:mlev) ) / REAL ( lyrcnt )
         srcp     = SUM ( p  (sfc:mlev) ) / REAL ( lyrcnt )
         srctheta = Theta ( srctK, srcp/100. )
   
      END IF
   
      srcthetaeK = Thetae ( srctK, srcp/100.0, srcrh, srcw )
   
      !~ Calculate temperature and pressure of the LCL
      !  ---------------------------------------------
      tlclK = TLCL ( tK(srclev), rh(srclev) )
      plcl  = p(srclev) * ( (tlclK/tK(srclev))**(Cp/Rd) )
   
      !~ Now lift the parcel
      !  -------------------
   
      buoy  = REAL ( 0 )
      pw    = srcw
      wflag = .false.
      DO k  = srclev, nz
         IF ( p (k) <= plcl ) THEN
   
            !~ The first level after we pass the LCL, we're still going to
            !~ lift the parcel dry adiabatically, as we haven't added the
            !~ the required code to switch between the dry adiabatic and moist
            !~ adiabatic cooling.  Since the dry version results in a greater
            !~ temperature loss, doing that for the first step so we don't over
            !~ guesstimate the instability.
            !  ----------------------------------------------------------------
   
            IF ( wflag ) THEN
               flag  = .false.
   
               !~ Above the LCL, our parcel is now undergoing moist adiabatic
               !~ cooling.  Because of the latent heating being undergone as
               !~ the parcel rises above the LFC, must iterative solve for the
               !~ parcel temperature using equivalant potential temperature,
               !~ which is conserved during both dry adiabatic and
               !~ pseudoadiabatic displacements.
               !  --------------------------------------------------------------
               ptK   = The2T ( srcthetaeK, p(k), flag )
   
               !~ Calculate the parcel mixing ratio, which is now changing
               !~ as we condense moisture out of the parcel, and is equivalent
               !~ to the saturation mixing ratio, since we are, in theory, at
               !~ saturation.
               !  ------------------------------------------------------------
               pw = SaturationMixingRatio ( ptK, p(k) )
   
               !~ Now we can calculate the virtual temperature of the parcel
               !~ and the surrounding environment to assess the buoyancy.
               !  ----------------------------------------------------------
               ptvK  = VirtualTemperature ( ptK, pw )
               tvK   = VirtualTemperature ( tK (k), w (k) )
   
               !~ Modification to account for water loading
               !  -----------------------------------------
               freeze = 0.033 * ( 263.15 - pTvK )
               IF ( freeze > 1.0 ) freeze = 1.0
               IF ( freeze < 0.0 ) freeze = 0.0
   
               !~ Approximate how much of the water vapor has condensed out
               !~ of the parcel at this level
               !  ---------------------------------------------------------
               freeze = freeze * 333700.0 * ( srcw - pw ) / 1005.7
   
               pTvK = pTvK - pTvK * ( srcw - pw ) + freeze
               dTvK ( k ) = ptvK - tvK
               buoy ( k ) = g * ( dTvK ( k ) / tvK )
   
            ELSE
   
               !~ Since the theta remains constant whilst undergoing dry
               !~ adiabatic processes, can back out the parcel temperature
               !~ from potential temperature below the LCL
               !  --------------------------------------------------------
               ptK   = srctheta / ( 100000.0/p(k) )**(Rd/Cp)
   
               !~ Grab the parcel virtual temperture, can use the source
               !~ mixing ratio since we are undergoing dry adiabatic cooling
               !  ----------------------------------------------------------
               ptvK  = VirtualTemperature ( ptK, srcw )
   
               !~ Virtual temperature of the environment
               !  --------------------------------------
               tvK   = VirtualTemperature ( tK (k), w (k) )
   
               !~ Buoyancy at this level
               !  ----------------------
               dTvK ( k ) = ptvK - tvK
               buoy ( k ) = g * ( dtvK ( k ) / tvK )
   
               wflag = .true.
   
            END IF
   
         ELSE
   
            !~ Since the theta remains constant whilst undergoing dry
            !~ adiabatic processes, can back out the parcel temperature
            !~ from potential temperature below the LCL
            !  --------------------------------------------------------
            ptK   = srctheta / ( 100000.0/p(k) )**(Rd/Cp)
   
            !~ Grab the parcel virtual temperture, can use the source
            !~ mixing ratio since we are undergoing dry adiabatic cooling
            !  ----------------------------------------------------------
            ptvK  = VirtualTemperature ( ptK, srcw )
   
            !~ Virtual temperature of the environment
            !  --------------------------------------
            tvK   = VirtualTemperature ( tK (k), w (k) )
   
            !~ Buoyancy at this level
            !  ---------------------
            dTvK ( k ) = ptvK - tvK
            buoy ( k ) = g * ( dtvK ( k ) / tvK )
   
         END IF

         !~ Chirp
         !  -----
  !          WRITE ( *,'(I15,6F15.3)' )k,p(k)/100.,ptK,pw*1000.,ptvK,tvK,buoy(k)
   
      END DO
   
      !~ Add up the buoyancies, find the LFC
      !  -----------------------------------
      flag   = .false.
      lfclev = -1
      nbuoy  = REAL ( 0 )
      pbuoy = REAL ( 0 )
      DO k = sfc + 1, nz
         IF ( tK (k) < 253.15 ) EXIT
         CAPE = CAPE + MAX ( buoy (k), 0.0 ) * ( hgt (k) - hgt (k-1) )
         CIN  = CIN  + MIN ( buoy (k), 0.0 ) * ( hgt (k) - hgt (k-1) )
   
         !~ If we've already passed the LFC
         !  -------------------------------
         IF ( flag .and. buoy (k) > REAL (0) ) THEN
            pbuoy = pbuoy + buoy (k)
         END IF
   
         !~ We are buoyant now - passed the LFC
         !  -----------------------------------
         IF ( .not. flag .and. buoy (k) > REAL (0) .and. p (k) < plcl ) THEN
            flag = .true.
            pbuoy = pbuoy + buoy (k)
            lfclev = k
         END IF
   
         !~ If we think we've passed the LFC, but encounter a negative layer
         !~ start adding it up.
         !  ----------------------------------------------------------------
         IF ( flag .and. buoy (k) < REAL (0) ) THEN
            nbuoy = nbuoy + buoy (k)

            !~ If the accumulated negative buoyancy is greater than the
            !~ positive buoyancy, then we are capped off.  Got to go higher
            !~ to find the LFC. Reset positive and negative buoyancy summations
            !  ----------------------------------------------------------------
            IF ( ABS (nbuoy) > pbuoy ) THEN
               flag   = .false.
               nbuoy  = REAL ( 0 )
               pbuoy  = REAL ( 0 )
               lfclev = -1
            END IF
         END IF

      END DO

      !~ Calculate lifted index by interpolating difference between
      !~ parcel and ambient Tv to 500mb.
      !  ----------------------------------------------------------
      DO k = sfc + 1, nz

         pm = 50000.
         pu = p ( k )
         pd = p ( k - 1 )

         !~ If we're already above 500mb just set lifted index to 0.
         !~ --------------------------------------------------------
         IF ( pd .le. pm ) THEN
            lidx = 0.
            EXIT
   
         ELSEIF ( pu .le. pm .and. pd .gt. pm) THEN

            !~ Found trapping pressure: up, middle, down.
            !~ We are doing first order interpolation.  
            !  ------------------------------------------
            lidxu = -dTvK ( k ) * ( pu / 100000. ) ** (Rd/Cp)
            lidxd = -dTvK ( k-1 ) * ( pd / 100000. ) ** (Rd/Cp)
            lidx = ( lidxu * (pm-pd) + lidxd * (pu-pm) ) / (pu-pd)
            EXIT

         ENDIF

      END DO
   
      !~ Assuming the the LFC is at a pressure level for now
      !  ---------------------------------------------------
      IF ( lfclev > 0 ) THEN
         PLFC = p   ( lfclev )
         ZLFC = hgt ( lfclev )
      END IF
   
      IF ( PLFC /= PLFC .OR. PLFC < REAL (0) ) THEN
         PLFC = REAL ( -1 )
         ZLFC = REAL ( -1 )
      END IF
   
      IF ( CAPE /= CAPE ) cape = 0.
   
      IF ( CIN  /= CIN  ) cin  = 0.

      !~ Chirp
      !  -----
  !       WRITE ( *,* ) ' CAPE: ', cape, ' CIN:  ', cin
  !       WRITE ( *,* ) ' LFC:  ', ZLFC, ' PLFC: ', PLFC
  !       WRITE ( *,* ) ''
  !       WRITE ( *,* ) ' Exiting buoyancy.'
  !       WRITE ( *,* ) ' ==================================== '
  !       WRITE ( *,* ) ''
   
    RETURN

!  END FUNCTION var_cape_afwa
  END SUBROUTINE var_cape_afwa

! ---- END modified from module_diag_afwa.F ---- !

!  FUNCTION var_cllmh(clfra, p, dz) [L. Fita Not working as function ?!]
  SUBROUTINE var_cllmh(clfra, p, dz, cllmh)
! Function to compute cllmh on a 1D column 1: low-cloud; 2: medium-cloud; 3: high-cloud [1]

    USE module_wrf_error
    USE module_model_constants

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, DIMENSION(dz), INTENT(in)                      :: clfra, p
    REAL, DIMENSION(3), INTENT(out)                      :: cllmh

! _ocal
    INTEGER                                              :: iz
    REAL(r_k)                                            :: zclearl, zcloudl, zclearm, zcloudm,       &
      zclearh, zcloudh
    REAL(r_k), DIMENSION(3)                              :: Dcllmh

!!!!!!! Variables
! clfra: cloudfraction as 1D verical-column [1]
! p: pressure values of the column
    sfname = 'var_cllmh'

    message = "CORDEX computing '" // TRIM(sfname) // "' "
    CALL wrf_debug(1000,message)

    zclearl = oneRK
    zcloudl = zeroRK
    zclearm = oneRK
    zcloudm = zeroRK
    zclearh = oneRK
    zcloudh = zeroRK

    Dcllmh = oneRK

    DO iz=1, dz
      IF (p(iz) < prmhc) THEN
        Dcllmh(3) = Dcllmh(3)*(oneRK-MAX(clfra(iz),zcloudh))/(oneRK-MIN(zcloudh,oneRK-ZEPSEC))
        zcloudh = clfra(iz)
      ELSE IF ( (p(iz) >= prmhc) .AND. (p(iz) < prmlc) ) THEN
        Dcllmh(2) = Dcllmh(2)*(oneRK-MAX(clfra(iz),zcloudm))/(oneRK-MIN(zcloudm,oneRK-ZEPSEC))
        zcloudm = clfra(iz)
      ELSE IF (p(iz) >= prmlc) THEN
        Dcllmh(1) = Dcllmh(1)*(oneRK-MAX(clfra(iz),zcloudl))/(oneRK-MIN(zcloudl,oneRK-ZEPSEC))
        zcloudl = clfra(iz)
      ELSE
        PRINT *,'  ' // TRIM(sfname) // ': This is weird, pressure:', p(iz), ' Pa fails out!!'
        PRINT *,'    from high, low cloud pressures:', prmhc, ' ,', prmlc,' Pa at z-level:', iz
        PRINT *,'    p_high > p:', prmhc,'> ',p(iz),' Pa'
        PRINT *,'    p_low > p >= p_high:', prmlc,'> ',p(iz),' >=', prmhc,' Pa'
        PRINT *,'    p_low >= p:', prmlc,'>= ',p(iz),' Pa'
        message = "program wrf: error on computing 'cllmh' for CORDEX"
        CALL WRF_ERROR_FATAL ( message )
        !STOP
      END IF
    END DO

    cllmh = REAL(oneRK - Dcllmh)*100.

    RETURN 

!  END FUNCTION var_cllmh
  END SUBROUTINE var_cllmh

!  REAL FUNCTION var_clt(clfra, dz)
  SUBROUTINE var_clt(clfra, dz, clt)
! Function to compute the total cloud following 'newmicro.F90' from LMDZ using 1D vertical 
!   column values

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, DIMENSION(dz), INTENT(in)                      :: clfra
    REAL, INTENT(out)                                    :: clt


! Local
    INTEGER                                              :: iz
    REAL(r_k)                                            :: zclear, zcloud, Dclt

!!!!!!! Variables
! cfra: 1-column cloud fraction values

    sfname = 'var_clt'

    message = "CORDEX computing '" // TRIM(sfname) // "' "
    CALL wrf_debug(1000,message)

    zclear = oneRK
    zcloud = zeroRK

    DO iz=1,dz
      zclear = zclear*(oneRK-MAX(clfra(iz),zcloud))/(oneRK-MIN(zcloud,1.-ZEPSEC))
      Dclt = oneRK - zclear
      zcloud = clfra(iz)
    END DO

   clt = REAL(Dclt)*100.

    RETURN

!  END FUNCTION var_clt
  END SUBROUTINE var_clt

  SUBROUTINE var_hur(t, press, q, dz, hur)
! Subroutine to compute relative humidity using August-Roche-Magnus approximation [1]

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, DIMENSION(dz), INTENT(in)                      :: t, press, q
    REAL, DIMENSION(dz), INTENT(out)                     :: hur

! Local
    INTEGER                                              :: k
    REAL                                                 :: tC, es, ws

!!!!!!! Variables
! t: temperature [K]
! press: pressure [Pa]
! q: mixing ratio [kgkg-1]
! dz: vertical extension
! hur: relative humidity [1]

    sfname = 'var_hur'

    ! August - Roche - Magnus formula (Avoiding overflow on last level)
    DO k=1,dz-1
      tC = t(k) - SVPT0
    
      es = ARM1 * exp(ARM2*tC/(tC+ARM3))
      ! Saturated mixing ratio
      ws = mol_watdry*es/(0.01*press(k)-es)

      ! Relative humidity
      hur(k) = q(k) / ws
    END DO

    RETURN

  END SUBROUTINE var_hur

  SUBROUTINE var_hurs(tas, ps, qas, hurs)
! Subroutine to compute relative humidity at 2m using August-Roche-Magnus approximation [1]

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: tas, ps, qas
    REAL, INTENT(out)                                    :: hurs

! Local
    REAL                                                 :: es, ws

!!!!!!! Variables
! tas: 2m temperature [K]
! ps: surface pressure [Pa]
! qas: 2m mixing ratio [kgkg-1]
! hurs: 2m relative humidity [1]

    sfname = 'var_hurs'

    ! August - Roche - Magnus formula
    es = ARM1 * exp(ARM2*(tas-SVPT0)/((tas-SVPT0)+ARM3))
    ! Saturated mixing ratio (assuming ps similar press_2m)
    ws = mol_watdry*es/(0.01*ps-es)

    ! Relative humidity
    hurs = qas / ws

    RETURN

  END SUBROUTINE var_hurs

  SUBROUTINE var_hus(q, dz, hus)
! Subroutine to compute specific humidity [1]

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, DIMENSION(dz), INTENT(in)                      :: q
    REAL, DIMENSION(dz), INTENT(out)                     :: hus

!!!!!!! Variables
! q: mixing ratio [kgkg-1]
! dz: vertical extension
! hus: specific humidity [1]

    sfname = 'var_hus'

    ! Specifc humidity
    hus = q / (1. + q)

    RETURN

  END SUBROUTINE var_hus

  SUBROUTINE var_huss(qas, huss)
! Subroutine to compute 2m specific humidity [1]

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: qas
    REAL, INTENT(out)                                    :: huss

!!!!!!! Variables
! qas: 2m mixing ratio [kgkg-1]
! huss: 2m specific humidity [1]

    sfname = 'var_huss'

    ! Specifc humidity
    huss = qas / (1. + qas)

    RETURN

  END SUBROUTINE var_huss

  SUBROUTINE var_massvertint(var, mutot, dz, deta, integral)
    ! Subroutine to vertically integrate a 1D variable in eta vertical coordinates

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, INTENT(in)                                     :: mutot
    REAL, DIMENSION(dz), INTENT(in)                      :: var, deta
    REAL, INTENT(out)                                    :: integral

! Local
    INTEGER                                              :: k

!!!!!!! Variables
! var: vertical variable to integrate (assuming kgkg-1)
! mutot: total dry-air mass in column
! dz: vertical dimension
! deta: eta-levels difference between full eta-layers

    sfname = 'var_massvertint'

!    integral=0.
!    DO k=1,dz
!      integral = integral + var(k)*deta(k)
!    END DO
     integral = SUM(var*deta)

    integral=integral*mutot/g

    RETURN

  END SUBROUTINE var_massvertint

  SUBROUTINE var_vertint(var, dz, deta, zweight, integral)
    ! Subroutine to vertically integrate a 1D variable in any vertical coordinates

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, DIMENSION(dz), INTENT(in)                      :: var, deta, zweight
    REAL, INTENT(out)                                    :: integral

!!!!!!! Variables
! var: vertical variable to integrate
! dz: vertical dimension
! deta: eta-levels difference between layers
! zweight: weight for each level to be applied (=1. for no effect)

    sfname = 'var_vertint'

    integral = SUM(var*deta*zweight)

    RETURN

  END SUBROUTINE var_vertint

  SUBROUTINE var_psl_ecmwf(PRPRESS, hgt, PTB, PRESBH, PRESBF, psl)
    ! Subroutine to compute sea level pressure using ECMWF method following Mats Hamrud and Philippe Courtier
    !   method found in LMDZ in phylmd/pppmer.F90 in combination with phylmd/ctsar.F90

!        IMPLICIT ARGUMENTS :  CONSTANTS FROM YOMCST,YOMGEM,YOMSTA.
!        --------------------

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: PRPRESS, hgt, PTB, PRESBH, PRESBF 
    REAL, INTENT(out)                                    :: psl

! Local
    REAL                                                 :: ghgt, PTSTAR, PT0, ZTSTAR
    REAL                                                 :: ZALPHA, POROG
    REAL                                                 :: ZDTDZSG, ZOROG, ZT0, ZTX, ZTY, ZX, ZY, ZY2
    REAL, PARAMETER                                      :: RDTDZ1 = -gammav

!!!!!!! Variables
! PRPRESS: Surface pressure [Pa]
! hgt: Terrain height [m]
! PTB: Temperature first half-level [K]
! PRESBH: Pressure first half-level [Pa]
! PRESBF: Pressure second full-level [Pa]
! psl: sea-level pressure

    sfname = 'var_psl_ecmwf'
    
    ! Height by gravity
    POROG = hgt*g

    !* 1. COMPUTES SURFACE TEMPERATURE
    !*   THEN STANDARD SURFACE TEMPERATURE.

    ZDTDZSG=-RDTDZ1/g
    ZALPHA=ZDTDZSG*r_d

    PTSTAR=PTB*(1.0+ZALPHA*(PRESBH/PRESBF-1.0))
    PT0=PTSTAR+ZDTDZSG*POROG

    !* 2.    POST-PROCESS MSL PRESSURE.
    !  --------------------------

    !* 2.1   COMPUTATION OF MODIFIED ALPHA AND TSTAR.

    ZTX=290.5
    ZTY=255.0

    IF (PTSTAR < ZTY) THEN
      ZTSTAR=0.5*(ZTY+PTSTAR)
    ELSEIF (PTSTAR < ZTX) THEN
      ZTSTAR=PTSTAR
    ELSE
      ZTSTAR=0.5*(ZTX+PTSTAR)
    ENDIF

    ZT0=ZTSTAR+ZDTDZSG*POROG
    IF (ZTX > ZTSTAR .AND. ZT0 > ZTX) THEN
      ZT0=ZTX
    ELSEIF (ZTX <= ZTSTAR .AND. ZT0 > ZTSTAR) THEN
      ZT0=ZTSTAR
    ELSE
      ZT0=PT0
    ENDIF

    ZOROG=SIGN(MAX(1.0,ABS(POROG)),POROG)
    ZALPHA=r_d*(ZT0-ZTSTAR)/ZOROG

    !* 2.2   COMPUTATION OF MSL PRESSURE.

    IF (ABS(POROG) >= 0.001) THEN
      ZX=POROG/(r_d*ZTSTAR)
      ZY=ZALPHA*ZX
      ZY2=ZY*ZY

      psl=PRPRESS*EXP(ZX*(1.0-0.5*ZY+1.0/3.*ZY2))
    ELSE
      psl=PRPRESS
    ENDIF

    RETURN

  END SUBROUTINE var_psl_ecmwf

  SUBROUTINE var_psl_ptarget(press, ps, hgt, ta, qv, dz, ptarget, psl)
    ! Subroutine to compute sea level pressure using a target pressure. Similar to the Benjamin 
    !   and Miller (1990). Method found in p_interp.F90

     INTEGER, INTENT(in)                                           :: dz
     REAL, DIMENSION(dz), INTENT(in)                               :: press, ta, qv
     REAL, INTENT(in)                                              :: ps, hgt, ptarget
     REAL, INTENT(out)                                             :: psl

! Local
     INTEGER                                                       :: kin
     INTEGER                                                       :: kupper
     REAL                                                          :: dpmin, dp, tbotextrap,   &
       tvbotextrap, virtual
     ! Exponential related to standard atmosphere lapse rate r_d*gammav/g
     REAL, PARAMETER                                               :: expon=r_d*gammav/g

!!!!!!! Variables
! press: Atmospheric pressure [Pa]
! ps: surface pressure [Pa]
! hgt: surface height
! ta: temperature [K]
! qv: water vapor mixing ratio
! dz: number of vertical levels
! psl: sea-level pressure

     sfname = 'var_psl_ptarget'

     IF (hgt /= 0.) THEN

       ! target pressure to be used for the extrapolation [Pa] (defined in namelist.input)
       !   ptarget = 70000. default value

       ! Minimal distance between pressures [Pa]
       dpmin=1.e6

       psl=0.

!      We are below both the ground and the lowest data level.

!      First, find the model level that is closest to a "target" pressure
!        level, where the "target" pressure is delta-p less that the local
!        value of a horizontally smoothed surface pressure field.  We use
!        delta-p = 150 hPa here. A standard lapse rate temperature profile
!        passing through the temperature at this model level will be used
!        to define the temperature profile below ground.  This is similar
!        to the Benjamin and Miller (1990) method, using  
!        700 hPa everywhere for the "target" pressure.

      kupper = 0
      loop_kIN: DO kin=dz,1,-1
        kupper = kin
        dp=abs( press(kin) - ptarget )
        IF (dp .GT. dpmin) EXIT loop_kIN
        dpmin=min(dpmin,dp)
      ENDDO loop_kIN

      tbotextrap=ta(kupper)*(ps/ptarget)**expon
      ! L. Fita. iCIMA, Janury 2018
      ! Using AFWA's one
      tvbotextrap=virtualTemperature(tbotextrap,qv(kupper))
      IF (tvbotextrap == 0.) THEN
        WRITE(message,*)'  ' // TRIM(sfname) //': wrong virtual temperature:', tvbotextrap, '!!'
        CALL wrf_debug(0,message)
        WRITE(message,*)'press:', press(kupper), ' ps:', ps, ' hgt:', hgt, ' ta:', ta(kupper), ' qv:',&
          ' ptarget:', ptarget, ' kupper:', kupper, ' dz:', dz 
        CALL wrf_debug(0,message)
        Lmessage = ''
        DO kin=1,dz-1
          WRITE(RS,'(F10.3)')press(kin) - ptarget
          WRITE(IS,'(I3)')kin
          WRITE(Lmessage,*) TRIM(Lmessage) //  TRIM(IS) // ' :' // TRIM(RS) // ', '
        END DO
        WRITE(RS,'(F10.3)')press(dz)
        WRITE(IS,'(I3)')dz
        WRITE(Lmessage,*) TRIM(Lmessage) //  TRIM(IS) // ' :' // TRIM(RS)
        CALL wrf_debug(0,Lmessage)
      END IF

      psl = ps*((tvbotextrap+gammav*hgt)/tvbotextrap)**(1/expon)
      IF (psl -1 == psl) THEN
        WRITE(message,*)'  ' // TRIM(sfname) //': wrong psl:', psl, '!!'
        CALL wrf_debug(0,message)
        WRITE(message,*)'press:', press(kupper), ' ps:', ps, ' hgt:', hgt, ' ta:', ta(kupper), ' qv:',&
          ' ptarget:', ptarget, ' kupper:', kupper, ' dz:', dz 
        CALL wrf_debug(0,message)
        WRITE(message,*)'expon', expon, ' tvbotextrap:', tvbotextrap, ' gammav:', gammav
        CALL wrf_debug(0,message)
      END IF
    ELSE
      psl = ps
    END IF

    RETURN

  END SUBROUTINE var_psl_ptarget

  SUBROUTINE var_psl_shuell(hgt, ps, zlev1, qlev1, tlev1, psl)
    ! Subroutine to compute sea level pressure using the hydrostatic equation with the Shuell 
    !   correction and existing code from 'phys/module_diag_afwa.F'

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: hgt, ps, zlev1, qlev1, tlev1
    REAL, INTENT(out)                                    :: psl

! Local
    REAL                                                 :: tauvrt, tauvr_sfc, tau_sfc, tau_sl,       &
      tauvr_sl, tauavg, zsfc

    ! Inverse of gravity
    REAL, PARAMETER                                      :: gi=1./g
    ! Sea-level height
    REAL, PARAMETER                                      :: zsl=0.0
    ! control Tau
    REAL, PARAMETER                                      :: taucr=r_d*gi*290.66
    ! Constant
    REAL, PARAMETER                                      :: const=0.005*g/r_d


!!!!!!! Variables
! hgt: surface height [m]
! ps: surgface pressure [Pa]
! zlev1: height first level [m]
! qlev1: mixing ratio at first level [kgkg-1]
! tlev1: temperature at first level [K]
! psl: sea level pressure [Pa]

    sfname = 'var_psl_shuell'

    psl = ps
    zsfc = hgt

    IF (hgt /= 0.) THEN

!     Compute layer tau (virtual temp*rd/g)
      tauvrt = tlev1*(1.0+0.608*qlev1)

!     Compute tau at the ground (z=zsfc) and sea level (z=0) assuming a constant lapse rate of 
!       gammav=6.5deg/km
      tauvr_sfc = tauvrt + (zlev1 - zsfc)*gammav
      tau_sfc = tauvr_sfc*r_d*gi
      tauvr_sl  = tauvrt + (zlev1 - zsl)*gammav
      tau_sl  = tauvr_sl*r_d*gi
    
!     if need be apply Sheull correction
      IF ((tau_sl.GT.taucr) .AND. (tau_sfc.LE.taucr)) THEN
        tau_sl = taucr
      ELSEIF ((tau_sl.GT.taucr) .AND. (tau_sfc.GT.taucr)) THEN
        tau_sl = taucr-const*(tau_sfc-taucr)**2
      ENDIF

!     compute mean tau
      tauavg = 0.5*(tau_sl+tau_sfc)
   
!     compute sea level pressure
      psl = ps*EXP(hgt/tauavg)
    END IF

    RETURN

  END SUBROUTINE var_psl_shuell

  SUBROUTINE var_press(p, pb, dz, press)
! Subroutine to compute air pressure [Pa]

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, DIMENSION(dz), INTENT(in)                      :: p, pb
    REAL, DIMENSION(dz), INTENT(out)                     :: press

!!!!!!! Variables
! p: WRF air-pressure perturbation [Pa]
! pb: WRF air-pressure base-state [Pa]
! dz: vertical extension (half-levels)
! press: air pressure (half-levels) [Pa]

    sfname = 'var_press'

    press = p + pb

    RETURN

  END SUBROUTINE var_press

  SUBROUTINE var_ta(tpot, press, dz, ta)
! Subroutine to compute air temperature [K]

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, DIMENSION(dz), INTENT(in)                      :: tpot, press
    REAL, DIMENSION(dz), INTENT(out)                     :: ta

!!!!!!! Variables
! tpot: WRF potential air-temperature [K]
! press: air-pressure [Pa]
! dz: vertical extension (half-levels)
! ta: air temperature [K]

    sfname = 'var_ta'

    ta = (tpot + 300.)*(press/p0)**(rcp)

    RETURN

  END SUBROUTINE var_ta

  SUBROUTINE var_uava(u, v, sina, cosa, dz, ua, va)
! Subroutine to compute earth-rotated wind components [ms-1]

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, DIMENSION(dz), INTENT(in)                      :: u, v
    REAL, INTENT(in)                                     :: sina, cosa
    REAL, DIMENSION(dz), INTENT(out)                     :: ua, va

! Local
    INTEGER                                              :: k

!!!!!!! Variables
! u, v: WRF unstaggered wind components [ms-1]
! sina, cosa: local sine and cosine of map rotation [1]
! dz: vertical extension
! ua, va: earth-rotated wind speed components [ms-1]
    sfname = 'var_uava'

    DO k=1, dz
      ua(k) = u(k)*cosa - v(k)*sina
      va(k) = u(k)*sina + v(k)*cosa
    END DO

    RETURN

  END SUBROUTINE var_uava

  SUBROUTINE var_uasvas(u10, v10, sina, cosa, uas, vas)
! Subroutine to compute 10m earth-rotated wind components [ms-1]

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: u10, v10, sina, cosa
    REAL, INTENT(out)                                    :: uas, vas

!!!!!!! Variables
! u10, v10: WRF 10m wind components [ms-1]
! sina, cosa: local sine and cosine of map rotation [1]
! uas, vas: earth-rotated 2m wind speed components [ms-1]

    sfname = 'var_uasvas'

    uas = u10*cosa - v10*sina
    vas = u10*sina + v10*cosa

    RETURN

  END SUBROUTINE var_uasvas

  SUBROUTINE var_zg(ph, phb, dz1, dz, zg, unzg)
! Subroutine to compute geopotential height [m]

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz, dz1
    REAL, DIMENSION(dz1), INTENT(in)                     :: ph, phb
    REAL, DIMENSION(dz1), INTENT(out)                    :: zg
    REAL, DIMENSION(dz), INTENT(out)                     :: unzg

! Local
    INTEGER                                              :: k

!!!!!!! Variables
! ph: WRF geopotential perturbation [m2s-2]
! phb: WRF geopotential base-state [m2s-2]
! dz1: vertical extension (full-levels)
! dz: vertical extension (half-levels)
! zg: geopotential height (full-levels) [m]
! unzg: geopotential height (half-levels) [m]

    sfname = 'var_zg'

    zg = (ph + phb) / g

    unzg(1:dz) = 0.5*(zg(1:dz) + zg(2:dz1))

    RETURN

  END SUBROUTINE var_zg

!!!!!!! Some more complex variables

  SUBROUTINE partial_agg(d1, vals, dvals, idvals, edvals, partagg)
! Subroutine to compute a partial aggregation of mass-values between an interval.

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: d1
    REAL, DIMENSION(d1), INTENT(in)                      :: vals, dvals
    REAL, INTENT(in)                                     :: idvals, edvals
    REAL, INTENT(out)                                    :: partagg

! Local
    INTEGER                                              :: k

!!!!!!! Variables
! d1: length of the axis
! vals: values to aggregate
! dvals: values along the axis
! idvals, edvals: initial and final value of the interval
! partagg: partial aggregation

    sfname = 'partial_agg'

    partagg = 0.

    DO k=1,d1
      IF (dvals(k) >= idvals .AND. (dvals(k) <= edvals )) partagg = partagg + vals(k)
    END DO

    RETURN

  END SUBROUTINE partial_agg

  SUBROUTINE interval_agg(d1, Ninterv, Ninterv1, vals, dvals, interval, intagg)
! Subroutine to compute aggregation by intervals along the full axis of of mass-values

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: d1, Ninterv, Ninterv1
    REAL, DIMENSION(d1), INTENT(in)                      :: vals, dvals
    REAL, DIMENSION(Ninterv), INTENT(in)                 :: interval
    REAL, DIMENSION(Ninterv1), INTENT(out)               :: intagg

! Local
    INTEGER                                              :: k, ijk
    REAL                                                 :: signd

!!!!!!! Variables
! d1: length of the axis
! Ninterv: Number of intervals
! Ninterv1: Number of aggregations (Ninterv+1)
! vals: values to aggregate
! dvals: values along the axis
! interval: values of the intervals
! intagg: aggregation along intervals

    sfname = 'interval_agg'

    intagg = 0.

    IF (dvals(1) > dvals(2)) THEN
      signd = -1
! Lest work fast ...
!      IF (interval(1) < interval(2)) THEN
!        WRITE(msg,*)'Axis values decrese, thus interval values must too'
!        CALL ErrMsg(msg, TRIM(sfname), -1)
!      END IF
    ELSE
      signd = 1
! Lest work fast ...
!      IF (interval(1) > interval(2)) THEN
!        WRITE(msg,*)'Axis values increase, thus interval values must too'
!        CALL ErrMsg(msg, TRIM(sfname), -1)
!      END IF
    END IF

    ijk = 1
    IF (signd == -1) THEN
      ! Axis in descending sense
      DO k=1, d1
        IF (dvals(k) < interval(ijk)) THEN
          IF (ijk < Ninterv) THEN
            ijk = ijk + 1
            intagg(ijk) = intagg(ijk) + vals(k)
          ELSE
            intagg(Ninterv1) = intagg(Ninterv1) + vals(k)
          END IF
        ELSE
          intagg(ijk) = intagg(ijk) + vals(k)
        END IF  
      END DO

    ELSE
      ! Axis in ascending sense
      DO k=1, d1
        IF (dvals(k) > interval(ijk)) THEN
          IF (ijk < Ninterv) THEN
            ijk = ijk + 1
            intagg(ijk) = intagg(ijk) + vals(k)
          ELSE
            intagg(Ninterv1) = intagg(Ninterv1) + vals(k)
          END IF
        ELSE
          intagg(ijk) = intagg(ijk) + vals(k)
        END IF  
      END DO

    END IF

    RETURN

  END SUBROUTINE interval_agg

  SUBROUTINE moist_group(Nmoist, i_qv, i_qc, i_qr, i_qs, i_qi, i_qg, i_qh, qvval, qcval, qrval, qsval,&
    qival, qgval, qhval, qvarvals)
! Subroutine to group 1D values from each water-species to a single variable

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: Nmoist, i_qv, i_qc, i_qr, i_qs, i_qi,     &
      i_qg, i_qh
    REAL, INTENT(in)                                     :: qvval, qcval, qrval, qsval, qival, qgval, &
      qhval
    REAL, DIMENSION(Nmoist), INTENT(out)                 :: qvarvals

!!!!!!! Variables
! Nmoist: number of water species
! i_q[v/c/r/s/i/g/h]: index for water vapour, cloud, rain, snow, ice, graupel and hail
! q[v/c/r/s/i/g/h]val: individual values for water vapour, cloud, rain, snow, ice, graupel and hail
! qvarvals: 2D values of multiple water species

    sfname = 'moist_group'

    IF (MAXVAL((/ i_qv, i_qc, i_qr, i_qs, i_qi, i_qg, i_qh /)) > Nmoist) THEN
      WRITE(message,*) '  ' // TRIM(sfname) // ': Wrong number of water-species Nmoist:', Nmoist,     &
        ' for:', i_qv, i_qc, i_qr, i_qs, i_qi, i_qg, i_qh,' increase it !!'
      CALL wrf_error_fatal(message)
    END IF

    qvarvals(i_qv) = qvval
    qvarvals(i_qc) = qcval
    qvarvals(i_qr) = qrval
    qvarvals(i_qs) = qsval
    qvarvals(i_qi) = qival
    qvarvals(i_qg) = qgval
    qvarvals(i_qh) = qhval

    RETURN

  END SUBROUTINE moist_group

  SUBROUTINE moist_group2D(Nmoist, d1, i_qv, i_qc, i_qr, i_qs, i_qi, i_qg, i_qh, qvval, qcval, qrval, &
    qsval, qival, qgval, qhval, qvarvals)
! Subroutine to group 2D values from each water-species to a single variable

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: Nmoist, d1, i_qv, i_qc, i_qr, i_qs, i_qi, &
      i_qg, i_qh
    REAL, DIMENSION(d1), INTENT(in)                      :: qvval, qcval, qrval, qsval, qival, qgval, &
      qhval
    REAL, DIMENSION(d1,Nmoist), INTENT(out)              :: qvarvals

!!!!!!! Variables
! Nmoist: number of water species
! i_q[v/c/r/s/i/g/h]: index for water vapour, cloud, rain, snow, ice, graupel and hail
! q[v/c/r/s/i/g/h]val: individual values for water vapour, cloud, rain, snow, ice, graupel and hail
! qvarvals: 2D values of multiple water species

    sfname = 'moist_group2D'

    IF (MAXVAL((/ i_qv, i_qc, i_qr, i_qs, i_qi, i_qg, i_qh /)) > Nmoist) THEN
      WRITE(message,*) '  ' // TRIM(sfname) // ': Wrong number of water-species Nmoist:', Nmoist,     &
        ' for:', i_qv, i_qc, i_qr, i_qs, i_qi, i_qg, i_qh,' increase it !!'
      CALL wrf_error_fatal(message)
    END IF

    qvarvals(:,i_qv) = qvval(:)
    qvarvals(:,i_qc) = qcval(:)
    qvarvals(:,i_qr) = qrval(:)
    qvarvals(:,i_qs) = qsval(:)
    qvarvals(:,i_qi) = qival(:)
    qvarvals(:,i_qg) = qgval(:)
    qvarvals(:,i_qh) = qhval(:)

    RETURN

  END SUBROUTINE moist_group2D

  SUBROUTINE moist_redistribute(Nmoist, qvarvals, i_qv, i_qc, i_qr, i_qs, i_qi, i_qg, i_qh, qvval,    &
    qcval, qrval, qsval, qival, qgval, qhval)
! Subroutine to re-distribute a multi water-species 1D value to each specie

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: Nmoist, i_qv, i_qc, i_qr, i_qs, i_qi,     &
      i_qg, i_qh
    REAL, DIMENSION(Nmoist), INTENT(in)                  :: qvarvals
    REAL, INTENT(out)                                    :: qvval, qcval, qrval, qsval, qival, qgval, &
      qhval

!!!!!!! Variables
! Nmoist: number of water species
! qvarvals: 1D values of multiple water species
! i_q[v/c/r/s/i/g/h]: index for water vapour, cloud, rain, snow, ice, graupel and hail
! q[v/c/r/s/i/g/h]val: individual values for water vapour, cloud, rain, snow, ice, graupel and hail

    sfname = 'moist_redistribute'

    qvval = qvarvals(i_qv)
    qcval = qvarvals(i_qc)
    qrval = qvarvals(i_qr)
    qsval = qvarvals(i_qs)
    qival = qvarvals(i_qi)
    qgval = qvarvals(i_qg)
    qhval = qvarvals(i_qh)

    RETURN

  END SUBROUTINE moist_redistribute

  SUBROUTINE moist_redistribute2D(Nmoist, d1, qvarvals, i_qv, i_qc, i_qr, i_qs, i_qi, i_qg, i_qh,     &
    qvval, qcval, qrval, qsval, qival, qgval, qhval)
! Subroutine to re-distribute a multi water-species 2D value to each specie

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: Nmoist, d1, i_qv, i_qc, i_qr, i_qs, i_qi, &
      i_qg, i_qh
    REAL, DIMENSION(d1,Nmoist), INTENT(in)               :: qvarvals
    REAL, DIMENSION(d1), INTENT(out)                     :: qvval, qcval, qrval, qsval, qival, qgval, &
      qhval

!!!!!!! Variables
! Nmoist: number of water species
! qvarvals: 2D values of multiple water species
! i_q[v/c/r/s/i/g/h]: index for water vapour, cloud, rain, snow, ice, graupel and hail
! q[v/c/r/s/i/g/h]val: individual values for water vapour, cloud, rain, snow, ice, graupel and hail

    sfname = 'moist_redistribute2D'

    qvval(:) = qvarvals(:,i_qv)
    qcval(:) = qvarvals(:,i_qc)
    qrval(:) = qvarvals(:,i_qr)
    qsval(:) = qvarvals(:,i_qs)
    qival(:) = qvarvals(:,i_qi)
    qgval(:) = qvarvals(:,i_qg)
    qhval(:) = qvarvals(:,i_qh)

    RETURN

  END SUBROUTINE moist_redistribute2D

  SUBROUTINE water_budget(dz, Nmoist, mutot, deta, dt, dtqvar, qvarhadv, qvarzadv, pwqvar, fqvar, zqvar)
! Subroutine to compute accumulated water budget
!! Code from: Fita and Flaounas, QJRMS, 2018
!! After Jian et al., 2008, Wea. Forecasting, 23, 44-61

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz, Nmoist
    REAL, INTENT(in)                                     :: mutot
    REAL, DIMENSION(dz), INTENT(in)                      :: deta
    REAL, INTENT(in)                                     :: dt
    REAL, DIMENSION(dz,Nmoist), INTENT(in)               :: dtqvar, qvarhadv, qvarzadv
    REAL, DIMENSION(dz,Nmoist), INTENT(out)              :: pwqvar, fqvar, zqvar

! Local
    INTEGER                                              :: k, l
    REAL                                                 :: wrfhadvqvar, wrfzadvqvar
    
!!!!!!! Variables
! dz: vertical extent
! Nmoist: number of water species
! mutot: total dry-air pressure [Pa]
! deta: thickness of eta layers
! dt: time-step [s]
! dtqvar: tendenchy of all water species [kgkg-1s-1]
! qvarhadv: horizontal advection of all water species [kgkg-1s-1]
! qvarzadv: vertical advection of all water species [kgkg-1s-1]
! pwqvar: total tendency of all water species [mm]
! fqvar: horizontal advecton of all water species [mm]
! zqvar: vertical advecton of all water species [mm]

    sfname = 'water_budget'

    ! Initializing variables
    pwqvar = 0.
    fqvar = 0.
    zqvar = 0.

    ! Looping on the vertical
    vertlevels: DO k=1,dz

      ! Looping on species
      DO l=1, Nmoist

        ! total tendency
        pwqvar(k,l) = (1./g)*mutot*deta(k)*dtqvar(k,l)*dt

        ! horizontal advection
        fqvar(k,l) = deta(k)*qvarhadv(k,l)*mutot*dt/g   
        ! vertical advection
        ! original acz(i,j) =  wrfzadvqv(i,k,j)/dt
        zqvar(k,l) = deta(k)*qvarzadv(k,l)*mutot*dt/g

      END DO

    END DO vertlevels

  END SUBROUTINE water_budget

  SUBROUTINE gustwind_afwa(dz, dt, u10, v10, pr, zagl, u_phy, v_phy, sa, ca, ublend, vblend, gustij)
! Subroutine to compute wind gust following heavy precip methodology found in phys/module_diag_afwa.F diagnostics

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, INTENT(in)                                     :: dt, u10, v10, pr, sa, ca
    REAL, DIMENSION(dz), INTENT(in)                      :: zagl, u_phy, v_phy
    REAL, INTENT(out)                                    :: ublend, vblend
    INTEGER, INTENT(out)                                 :: gustij

! Local
    INTEGER                                              :: k
    REAL                                                 :: wind_vel, prate_mm_per_hr
    LOGICAL                                              :: is_target_level
    REAL                                                 :: factor, u1km, v1km
    REAL                                                 :: ublend0, vblend0

!!!!!!! Variables
! dz: number of vertical levels
! dt: time-step [s]
! u10, v10: 10-m horizontal wind components [ms-1]
! pr: time-step precipitation [kgm-2]
! zagl: height above ground [m]
! u_phy, v_phy: air wind speed [ms-1]
! sa, ca: : Map factors sinus and cosionus
! ublend, vblend: blended winds at 1 km AGL to be considered as wind gust components [ms-1]
! gustij: whether the grid point got gust winds (1) or not (0)

    sfname = 'gustwind_afwa'

    ! Calculate the max 10 m wind speed between output times
    ! ------------------------------------------------------
    ! UPDATE 20150112 - GAC
    ! Diagnose from model 10 m winds, and blend with 1 km AGL
    ! winds when precipitation rate is > 50 mm/hr to account
    ! for increased surface wind gust potential when precip
    ! is heavy and when winds aloft are strong.  Will use the
    ! higher of the surface and the blended winds. Blending
    ! is linear weighted between 50-150 mm/hr precip rates.
    ! -------------------------------------------------------
    ublend = 0.
    vblend = 0.

    wind_vel = SQRT(u10*u10 + v10*v10)
    prate_mm_per_hr = (pr/dt)*3600.

    ! Is this an area of heavy precip?  Calculate 1km winds to blend down
    ! -------------------------------------------------------------------
    IF ( prate_mm_per_hr .GT. 50. ) THEN
      gustij = 1
      is_target_level = .FALSE.
      DO k=1,dz
        IF ( (zagl(k) >= 1000.) .AND. (.NOT.is_target_level) .AND. (k .ne. 1)) THEN
          is_target_level = .TRUE.
          u1km = u_phy(k-1) + (1000.-zagl(k-1))*((u_phy(k)-u_phy(k-1))/(zagl(k)))
          v1km = v_phy(k-1) + (1000.-zagl(k-1))*((v_phy(k)-v_phy(k-1))/(zagl(k)))
          EXIT ! We've found our level, break the loop
        ENDIF
      ENDDO
          
      ! Compute blended wind
      ! --------------------
      factor = MAX( ((150.-prate_mm_per_hr)/100.), 0.)
      ublend0 = u10*factor + u1km*(1.-factor)
      vblend0 = v10*factor + v1km*(1.-factor)

      ! Rotating winds
      ublend = ublend0*ca-vblend0*sa
      vblend = vblend0*ca+ublend0*sa
      WRITE(message,*)'   Lluis dz:', dz, ' dt:', dt, ' u10:', u10,' v10:', v10, ' pr:', pr, ' k:', k, ' zagl:', zagl(k), ' u_phy:',  u_phy(k), ' v_phy.', v_phy(k), ' sa:', sa, ' ca:', ca
      CALL wrf_debug(10, message)
      WRITE(message,*)'   Lluis factor:', factor, ' prate_mm_per_hr:', prate_mm_per_hr, ' u1km:', u1km,' v1km:', v1km, ' ublend0:', ublend0, ' vblend0:', vblend0, ' ublend:',  ublend, ' vblend.', vblend
      CALL wrf_debug(10, message)
    ELSE
      gustij = 0
      ublend = 0.
      vblend = 0.
    ENDIF

    RETURN

  END SUBROUTINE gustwind_afwa

! From clWRF heavy
  SUBROUTINE gustwind_Brasseur01(dz, geopoth, qr, lr, pt, tke, u, v, hpbl, topo, sina, cosa,          &
    ugustwind, vgustwind, gustij) 
! Subroutine to compute 1D wind gust following Brasseur (2001), MWR. 
! Downdraft effect from convection has not been applied

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, DIMENSION(dz), INTENT(in)                      :: geopoth, qr, lr, pt, tke, u, v
    REAL, INTENT(in)                                     :: hpbl, topo, sina, cosa
    REAL, INTENT(out)                                    :: ugustwind, vgustwind
    INTEGER, INTENT(out)                                 :: gustij
! Not needed for climate runs
!    REAL, DIMENSION, INTENT(out)                         :: ugustlow, vgustlow, ugustup, vgustup
  
! Local
    INTEGER                                              :: i,j,k,l, Nlevs, km2, ilev
    INTEGER                                              :: Ngustlevels, Ngustlowlev, maxlev
    REAL, DIMENSION(dz)                                  :: vpt, heights, dheights, dvpt
    REAL, DIMENSION(dz,2)                                :: wge !, wge_low, wge_up
    INTEGER, DIMENSION(dz)                               :: gustlevels
    REAL                                                 :: preshpbl, tke_int, vpt_int
    REAL                                                 :: height2pres, virtualpottemp, pres2height
    REAL                                                 :: wgemax !, wgemaxlow, wgemaxup
    REAL                                                 :: ugust, vgust
!    REAL                                                 :: ugustl, vgustl, ugustu, vgustu
    CHARACTER(LEN=1024)                                  :: outstring

!!!!!!!!!! Variables
! geopoth: 3D geopotential height
! qr: 3D vapour water mixing ratio
! lr: 3D liquid water mixing ratio
! pt: 3D potential temperature
! tke: 3D turbulent kinetic energy
! u, v: 3D wind components
! hpbl: pbl height
! topo: topography
! sina, cosa: Map factors sinus and cosionus
! ugustwind, vgustwind: gust wind components
! ugustlow, vgustlow: gust wind components of lower bounder
! ugustup, vgustup: gust wind components of upper bounder
! Nlevs: number of layers within pbl
! Ngustlevels: number of pbl layers with a int_tke >= int_ept
! Ngustlowlev: number of pbl layers with a 2.5*tke(lay)/11. >= int_ept
! vpt: virtual potential temperature
! heights: vector of heights according hydrostatic conversion from pressure
! dheights: vector of differences between heights
! wge: estimated wind gusts 
! wge_low: low doundaries of estimated wind gust
! wge_up: up boundary of estimated wind gust
! gustlevels: levels of pbl with a possible deflected air parcel
! preshpbl: hydrostatic equivalent pbl height pressure
! tke_int: tke integral up to a given layer
! vpt_int: vpt integral up to a given layer
! wgemax: maximum wge
! wgemaxlow: lower wge boundary
! wgemaxup: upper wge boundary
! zero: zero values
! gustij: whether the grid point got gust winds (1) or not (0)

!!!!!!!!!! Functions
! virtualpottemp: calculates virtual potential temperature

    sfname = 'gustwind_Brasseur01'

    km2=INT(dz/2)

    ugustwind=0.
    vgustwind=0.
!! Not needed on climate runs
!    ugustlow=0.
!    vgustlow=0.
!    ugustup=0.
!    vgustup=0.

    maxlev=0
    heights=0.
    dheights=0.
    vpt=0.
    wge=0.
    !wge_low=0.
    !wge_up=0.

    k=1  
    IF ((geopoth(k+1)/g-topo) <= hpbl .AND. (k <= dz)) THEN
      !!outstring  = '  ' // TRIM(sfname) // ' _______'
      !!CALL wrf_debug(750, outstring)
      !!WRITE(outstring,"(4x,A4,1x,11(A10,1x))")'k','geoph','topo','geo_h','hpbl','pt', 'qr','lr',      &
      !!  'vpt', 'u', 'v', 'h'
      CALL wrf_debug(750, outstring)
    END IF
!!       'DO WHILE' does not work
!!        DO WHILE (( (geopoth(k+1)/g-topo) <= hpbl) .AND. (k+1 <= dz) )
    DO k=1,dz-1
      IF ( (geopoth(k+1)/g-topo) > hpbl ) THEN
        EXIT
      ELSE
!! Function does not work. Let's make it with subroutines
!!       vpt(k)=virtualpottemp(pt(k), qr(k), lr(k))
        CALL VirtPotTemp(pt(k), qr(k), lr(k), vpt(k))
        heights(k)=geopoth(k)/g-topo

        !!WRITE(outstring,*)'    ',k, geopoth(k), topo, geopoth(k)/g-topo, hpbl, pt(k), qr(k), lr(k),   &
        !!  vpt(k), u(k), v(k), heights(k)
        !!CALL wrf_debug(750, outstring)
      END IF 
    END DO
    Nlevs=k-1
!
!! Computation of wind gusts only if more than 1 levels behind 'hpbl' are found
      
    more1lev: IF (Nlevs > 1) THEN

      !wge_up(1:Nlevs,1)=u(1:Nlevs)
      !wge_up(1:Nlevs,2)=v(1:Nlevs)

      !!WRITE(outstring,"(3x,A14,1x)")'Integrating...'
      !!CALL wrf_debug ( 750, TRIM(outstring) )

      dheights(1:Nlevs-1)=heights(2:Nlevs)-heights(1:Nlevs-1)
        
      Ngustlevels=0
      Ngustlowlev=0
      !!WRITE(outstring,"(9(A10,1x))")'ilev','vpt','dvpt','h','dh','tke','tkeInt','dvptInt',  '25tke11'
      !!CALL wrf_debug(750, outstring)

      dvpt=0.
      dvpt(1)=g*(vpt(2)-vpt(1))/vpt(1)
      dvpt(2:Nlevs-1)=g*(vpt(3:Nlevs)-vpt(1:Nlevs-2))/vpt(2:Nlevs-1)
      dvpt(Nlevs)=g*(vpt(Nlevs)-vpt(Nlevs-1))/vpt(Nlevs)
      ilev=1
      !!WRITE(outstring,*)ilev, vpt(ilev), dvpt(ilev), heights(ilev), dheights(ilev), tke(ilev), 0.0,   &
      !!      0.0, 2.5*tke(ilev)/11.
      !!CALL wrf_debug(750, outstring)

      DO ilev=2,Nlevs
        tke_int = 0.
        vpt_int = 0.

! Trapezoidal integrations
!        CALL NumIntegration(tke,heights,ilev,Nlevs,tke_int)
!        CALL NumIntegration(dvpt,heights,ilev,Nlevs,vpt_int)
! Standard integrations
        DO k=1,ilev
          tke_int = tke_int + tke(k)*dheights(k)
          vpt_int = vpt_int + dvpt(k)
        END DO

        tke_int=tke_int/heights(ilev)
        !!WRITE(outstring,*)ilev, vpt(ilev), dvpt(ilev), heights(ilev),  dheights(ilev), tke(ilev),     &
        !!  tke_int, vpt_int, 2.5*tke(ilev)/11.
        !!CALL wrf_debug(750, outstring)

        IF (tke_int >= vpt_int) THEN
          Ngustlevels=Ngustlevels+1
          gustlevels(Ngustlevels)=ilev
          wge(Ngustlevels,1)=u(ilev)
          wge(Ngustlevels,2)=v(ilev)
        ELSE
          ! End of deflection
          EXIT
        END IF

!        IF (2.5*tke(ilev)/11. >= vpt_int) THEN
!          Ngustlowlev=Ngustlowlev+1
!          wge_low(Ngustlowlev,1)=u(ilev)
!          wge_low(Ngustlowlev,2)=v(ilev)
!        END IF
      END DO

      !!WRITE(outstring,*)'  ' // TRIM(sfname) // ': Wind gust estimation...'
      !!CALL wrf_debug( 750, outstring)

! gustwind
!!
      IF (Ngustlevels > 0 ) THEN
        gustij = 1
        wgemax=MAXVAL(SQRT(wge(:,1)*wge(:,1)+wge(:,2)*wge(:,2)))
        DO ilev=1,Ngustlevels
          IF (ABS(wgemax - SQRT(wge(ilev,1)*wge(ilev,1)+wge(ilev,2)*wge(ilev,2))) < nullv) THEN
            maxlev=ilev
            EXIT
          END IF
        END DO
        ugust=wge(maxlev,1)
        vgust=wge(maxlev,2)
      ELSE
        gustij = 0
        ugust=0.
        vgust=0.
      ENDIF

      !!WRITE(outstring,"(5x,A14,1x,I3,1x,3(A10,1x))")'N gust levels:',Ngustlevels,'gust lev', 'u wge', &
      !!  'v wge'
      !!CALL wrf_debug (750, outstring)
      !!DO l=1,Ngustlevels
        !!WRITE(outstring,"(22x,I10,1x,2(f10.4,1x))")gustlevels(l),wge(l,1:2)
        !!CALL wrf_debug(750,outstring)
      !!END DO

! gustwind lower bound
!!
!      IF (Ngustlowlev > 0) THEN
!        wgemaxlow=MAXVAL(SQRT(wge_low(:,1)*wge_low(:,1)+wge_low(:,2)*wge_low(:,2)))
!        DO ilev=1,Ngustlowlev
!          IF (ABS(wgemaxlow - SQRT(wge_low(ilev,1)*wge_low(ilev,1)+wge_low(ilev,2)*wge_low(ilev,2)))  &
!            < zero) THEN
!            maxlev=ilev
!            EXIT
!          END IF
!        END DO
!        ugustl=wge_low(maxlev,1)
!        vgustl=wge_low(maxlev,2)          
!      ELSE
!        ugustl=0.
!        vgustl=0.
!      END IF

! gustwind upperbound
!!
!      IF (Nlevs > 0) THEN
!        wgemaxup=MAXVAL(SQRT(wge_up(:,1)*wge_up(:,1)+wge_up(:,2)*wge_up(:,2)))
!        DO ilev=1,Nlevs
!          IF (ABS(wgemaxup - SQRT(wge_up(ilev,1)*wge_up(ilev,1)+wge_up(ilev,2)*wge_up(ilev,2))) <     &
!            zero ) THEN
!            maxlev=ilev
!            EXIT
!          END IF
!        END DO
!        ugustu=wge_up(maxlev,1)
!        vgustu=wge_up(maxlev,2)
!      ELSE
!        ugustu=0.
!        vgustu=0.
!      END IF
    ELSE
      gustij = 0
      ugust=0.
      vgust=0.
!      ugustl=0.
!      vgustl=0.
!      ugustu=0.
!      vgustu=0.
    END IF more1lev

    ! Rotation of winds to Earth surface
#if (EM_CORE == 1)
    ugustwind = ugust*cosa-vgust*sina
    vgustwind = vgust*cosa+ugust*sina
!    ugustlow = ugustl*cosa-vgustl*sina
!    vgustlow = vgustl*cosa+ugustl*sina
!    ugustup = ugustu*cosa-vgustu*sina
!    vgustup = vgustu*cosa+ugustu*sina
#else
    ugustwind = ugust
    vgustwind = vgust
!    ugustlow = ugustl
!    vgustlow = vgustl
!    ugustup = ugustu
!    vgustup = vgustu
#endif

    ! Some debugging printing
    !!outstring = '    ' // TRIM(sfname) // ' _______'
    !!CALL wrf_debug(750, outstring)
    !!WRITE(outstring,"(5x,6(A12,1x))")'name','u wind','v wind', 'rot. u wind', 'rot. v wind','wind'
    !!CALL wrf_debug(750, outstring)
    !!WRITE(outstring,"(11x,A12,1x,5(f12.5,1x))")'Gust wind', ugust, vgust, ugustwind, vgustwind,       &
    !!  sqrt(ugustwind*ugustwind+vgustwind*vgustwind)
!    CALL wrf_debug(750, outstring)
!    WRITE(outstring,"(11x,A12,1x,5(f12.5,1x))")'Low bound', ugustl, vgustl, ugustlow, vgustlow,       &
!      sqrt(ugustlow*ugustlow+vgustlow*vgustlow)
!    CALL wrf_debug(750, outstring)
!    WRITE(outstring,"(11x,A12,1x,5(f12.5,1x))")'Upper bound', ugustu, vgustu, ugustup, vgustup,       &
!      sqrt(ugustup*ugustup+vgustup*vgustup)
!    CALL wrf_debug(750, outstring)

    RETURN

  END SUBROUTINE gustwind_Brasseur01

  SUBROUTINE NumIntegration(yvals, xvals, Nvalint, Ntotval, intval)
! Subroutine to compute numerical integrations according to the trapezoidal methodology

    IMPLICIT NONE
  
    REAL, DIMENSION(Ntotval), INTENT(in)                 :: yvals, xvals
    INTEGER, INTENT(in)                                  :: Nvalint, Ntotval
    REAL, INTENT(out)                                    :: intval
  
!!!!!!! Variables
! yvals: values to integrate
! xvals: equivalent positions on x-axis
! Nvalint: range of the integral
! Ntotval: total number of values
! intval: result of the integration

    intval=0.
    intval=SUM((xvals(2:Nvalint)-xvals(1:Nvalint-1))*(yvals(1:Nvalint-1)+yvals(2:Nvalint))/2.)

    RETURN
  
  END SUBROUTINE NumIntegration

  REAL FUNCTION virtualpottemp(pt_vpt, r_vpt, l_vpt)
! Function to compute virtual potential temperature

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: pt_vpt, r_vpt, l_vpt 
! Local
    REAL, PARAMETER                                      :: vptA=0.61

    virtualpottemp=pt_vpt*(1+vptA*r_vpt-l_vpt)

    RETURN

  END FUNCTION virtualpottemp

  SUBROUTINE VirtPotTemp(pt_vpt, r_vpt, l_vpt, vpt)
! Subroutine to compute virtual potential temperature

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: pt_vpt, r_vpt, l_vpt 
    REAL, INTENT(out)                                    :: vpt
! Local
    REAL, PARAMETER                                      :: vptA=0.61

    vpt=pt_vpt*(1.+vptA*r_vpt-l_vpt)

    RETURN

  END SUBROUTINE VirtPotTemp

  SUBROUTINE var_zmla_generic(dz, qv, tpot, z, topo, dqvar, refdt, zmla)
!  Subroutine to compute pbl-height following a generic method
!    from Nielsen-Gammon et al., 2008 J. Appl. Meteor. Clim.
!    applied also in Garcia-Diez et al., 2013, QJRMS
!   where 
!     "The technique identifies the ML height as a threshold increase of
!     potential temperature from 
!       its minimum value within the boundary layer."
!   here applied similarly to Garcia-Diez et al. where 
!      zmla = "...first level where potential temperature exceeds the minimum
!      potential temperature
!        reached in the mixed layer by more than 1.5 K"

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: dz
    REAL, DIMENSION(dz), INTENT(in)                      :: qv, tpot, z
    REAL, INTENT(in)                                     :: topo, dqvar, refdt 
    REAL, INTENT(out)                                    :: zmla

! Local
    INTEGER                                              :: mldlev, bllev
    REAL                                                 :: tpotmin

!!!!!!! Variables
! qv: water vapour mixing ratio
! tpot: potential temperature [K]
! z: height above sea level [m]
! topo: topographic height [m]
! dqvar: Pecentage of difference of mixing ratio used to determine Mixed layer depth
! refdt: Change [K] in temperature to determine boundary layer height
! zmla: boundary layer height [m]

    sfname = 'var_zmla_generic'

    ! MLD = Mixed layer with no substantial variation of mixing ratio /\qv < 10%
    ! ?
    !PRINT *,'  Mixed layer mixing ratios qv[1] lev qv[lev] dqvar% _______'
    DO mldlev = 2, dz
      IF (ABS(qv(mldlev)-qv(1))/qv(1) > dqvar ) EXIT
    !  PRINT *,qv(1), mldlev, qv(mldlev), ABS(qv(mldlev)-qv(1))/qv(1)
    END DO

    ! Looking for the minimum potential temperature within the MLD [tpotmin =
    ! min(tpot)_0^MLD]
    tpotmin = MINVAL(tpot(1:mldlev))

    ! Determine the first level where tpot > tpotmin + 1.5 K
    !PRINT *,'  Mixed layer tpotmin lev tpotmin[lev] dtpot _______'
    DO bllev = 1, dz
      IF (ABS(tpot(bllev)-tpotmin) > refdt ) EXIT
    !  PRINT *,tpotmin, bllev, tpot(bllev), ABS(tpot(bllev)-tpotmin)
    END DO
    
    !PRINT *,'   height end MLD:', z(mldlev)
    !PRINT *,'   pbl height:', z(bllev)

    zmla = z(bllev) - topo

    RETURN

  END SUBROUTINE var_zmla_generic

  SUBROUTINE var_zwind(d1, u, v, z, u10, v10, sa, ca, newz, unewz, vnewz)
! Subroutine to extrapolate the wind at a given height following the 'power law' methodology
!    wss[newz] = wss[z1]*(newz/z1)**alpha
!    alpha = (ln(wss[z2])-ln(wss[z1]))/(ln(z2)-ln(z1))
! AFTER: Phd Thesis: 
!   Benedicte Jourdier. Ressource eolienne en France metropolitaine : methodes d’evaluation du 
!   potentiel, variabilite et tendances. Climatologie. Ecole Doctorale Polytechnique, 2015. French
!

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: d1
    REAL, DIMENSION(d1), INTENT(in)                      :: u,v,z
    REAL, INTENT(in)                                     :: u10, v10, sa, ca, newz
    REAL, INTENT(out)                                    :: unewz, vnewz

! Local
    INTEGER                                              :: inear
    REAL                                                 :: zaground
    REAL, DIMENSION(2)                                   :: v1, v2, zz, alpha, uvnewz

!!!!!!! Variables
! u,v: vertical wind components [ms-1]
! z: height above surface [m]
! u10,v10: 10-m wind components [ms-1]
! topo: topographical height [m]
! sa, ca: local sine and cosine of map rotation [1.]
! newz: desired height above grpund of extrapolation
! unewz,vnewz: Wind compoonents at the given height [ms-1]

    sfname = 'var_zwind'

    !!WRITE(message,*)' ilev zaground newz z[ilev+1] z[ilev+2] _______'
    !!CALL wrf_debug(750,message)
    ! Looking for the level  below desired height
    IF (z(1) < newz ) THEN
      DO inear = 1,d1-2
        ! L. Fita, CIMA. Feb. 2018
        !! Choose between extra/inter-polate. Maybe better interpolate?
        ! Here we extrapolate from two closest lower levels
        !zaground = z(inear+2)
        ! Here we interpolate between levels
        zaground = z(inear+1)
        !!WRITE(message,*)inear, z(inear), newz, z(inear+1), z(inear+2)
        !!CALL wrf_debug(750,message)
        IF ( zaground >= newz) EXIT
      END DO
    ELSE
      !!WRITE(message,*)1, z(1), newz, z(2), z(3), ' z(1) > newz'
      !!CALL wrf_debug(750,message)
      inear = d1 - 2
    END IF

    IF (inear == d1-2) THEN
    ! No vertical pair of levels is below newz, using 10m wind as first value
    ! and the first level as the second
       v1(1) = u10
       v1(2) = v10
       v2(1) = u(1)
       v2(2) = v(1)
       zz(1) = 10.
       zz(2) = z(1)
    ELSE
       v1(1) = u(inear)
       v1(2) = v(inear)
       v2(1) = u(inear+1)
       v2(2) = v(inear+1)
       zz(1) = z(inear)
       zz(2) = z(inear+1)
    END IF

    ! Computing for each component
    alpha = (LOG(ABS(v2))-LOG(ABS(v1)))/(LOG(zz(2))-LOG(zz(1)))
    !!WRITE(message,*)' Computing with v1:', v1, ' ms-1 v2:', v2, ' ms-1'
    !!CALL wrf_debug(750,message)
    !!WRITE(message,*)' z1:', zz(1), 'm z2:', zz(2), ' m'
    !!CALL wrf_debug(750,message)
    !!WRITE(message,*)' alhpa u:', alpha(1), ' alpha 2:', alpha(2)
    !!CALL wrf_debug(750,message)

    uvnewz = v1*(newz/zz(1))**alpha
    ! Earth-rotation
    unewz = uvnewz(1)*ca - uvnewz(2)*sa
    vnewz = uvnewz(1)*sa + uvnewz(2)*ca

    !!WRITE(message,*)'  result vz:', uvnewz
    !!CALL wrf_debug(750,message)

    RETURN

  END SUBROUTINE var_zwind

  SUBROUTINE var_zwind_log(d1, u, v, z, u10, v10, sa, ca, newz, unewz, vnewz)
! Subroutine to extrapolate the wind at a given height following the 'logarithmic law' methodology
!    wsz = wss[z2]*(ln(newz)-ln(z0))(ln(z2)-ln(z0))
!    ln(z0) = (ws(z2)*ln(z1)-ws(z1)*ln(z2))/(ws(z2)-ws(z1))
! AFTER: Phd Thesis: 
!   Benedicte Jourdier. Ressource eolienne en France metropolitaine : methodes d’evaluation du 
!   potentiel, variabilite et tendances. Climatologie. Ecole Doctorale
!   Polytechnique, 2015. French
!
    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: d1
    REAL, DIMENSION(d1), INTENT(in)                      :: u,v,z
    REAL, INTENT(in)                                     :: u10, v10, sa, ca, newz
    REAL, INTENT(out)                                    :: unewz, vnewz

! Local
    INTEGER                                              :: inear
    REAL                                                 :: zaground
    REAL, DIMENSION(2)                                   :: v1, v2, zz, logz0, uvnewz

!!!!!!! Variables
! u,v: vertical wind components [ms-1]
! z: height above surface on half-mass levels [m]
! u10,v10: 10-m wind components [ms-1]
! sa, ca: local sine and cosine of map rotation [1.]
! newz: desired height above grpund of extrapolation
! unewz,vnewz: Wind compoonents at the given height [ms-1]

    sfname = 'var_zwind_log'

    IF (z(1) < newz ) THEN
      DO inear = 1,d1-2
        ! L. Fita, CIMA. Feb. 2018
        !! Choose between extra/inter-polate. Maybe better interpolate?
        ! Here we extrapolate from two closest lower levels
        !zaground = z(inear+2)
        zaground = z(inear+1)
        IF ( zaground >= newz) EXIT
      END DO
    ELSE
      inear = d1 - 2
    END IF

    IF (inear == d1-2) THEN
    ! No vertical pair of levels is below newz, using 10m wind as first value
    ! and the first level as the second
       v1(1) = u10
       v1(2) = v10
       v2(1) = u(1)
       v2(2) = v(1)
       zz(1) = 10.
       zz(2) = z(1)
    ELSE
       v1(1) = u(inear)
       v1(2) = v(inear)
       v2(1) = u(inear+1)
       v2(2) = v(inear+1)
       zz(1) = z(inear)
       zz(2) = z(inear+1)
    END IF

    ! Computing for each component
    logz0 = (v2*LOG(zz(1))-v1*LOG(zz(2)))/(v2-v1)

    uvnewz = v2*(LOG(newz)-logz0)/(LOG(zz(2))-logz0)
    ! Earth-rotation
    unewz = uvnewz(1)*ca - uvnewz(2)*sa
    vnewz = uvnewz(1)*sa + uvnewz(2)*ca

    RETURN

  END SUBROUTINE var_zwind_log


  SUBROUTINE var_zwind_MOtheor(ust, znt, rmol, u10, v10, sa, ca, newz, uznew, vznew)
  ! Subroutine of wind extrapolation following Moin-Obukhov theory R. B. Stull, 1988, 
  !   Springer (p376-383)
  ! NOTE only usefull for newz < 80. m

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: ust, znt, rmol, u10, v10, sa, ca
    REAL, INTENT(in)                                     :: newz
    REAL, INTENT(out)                                    :: uznew, vznew

! Local
    REAL                                                 :: OL
    REAL                                                 :: stability
    REAL                                                 :: wsz, alpha
    REAL, DIMENSION(2)                                   :: uvnewz

!!!!!!! Variables
! ust: u* in similarity theory [ms-1]
! z0: roughness length [m]
!!! L. Fita, CIMA. Feb. 2018
!! NOT SURE if it should be z0 instead?
! znt: thermal time-varying roughness length [m]
! rmol: inverse of Obukhov length [m-1]
! u10: x-component 10-m wind speed [ms-1]
! v10: y-component 10-m wind speed [ms-1]
! sa, ca: local sine and cosine of map rotation [1.]
! 
    sfname = 'var_zwind_MOtheor'

    ! Obukhov Length (using the Boussinesq approximation giving Tv from t2)
    OL = 1/rmol

    ! Wind speed at desired height
    CALL stabfunc_businger(newz,OL,stability)
    wsz = ust/karman*( LOG(newz/znt) + stability)

    ! Without taking into account Ekcman pumping, etc... redistributed by components unsing 10-m wind
    !   as reference...
    alpha = ATAN2(v10,u10)
    uvnewz(1) = wsz*COS(alpha)
    uvnewz(2) = wsz*SIN(alpha)

    ! Earth-rotation
    uznew = uvnewz(1)*ca - uvnewz(2)*sa
    vznew = uvnewz(1)*sa + uvnewz(2)*ca

    RETURN

  END SUBROUTINE var_zwind_MOtheor

  ! L. Fita, CIMA. Feb. 2018
  ! WRF seems to have problems with my functions, let'suse subroutine instead
  !REAL FUNCTION stabfunc_businger(z,L)
  SUBROUTINE stabfunc_businger(z,L,stabfunc_busingerv)
  ! Fucntion of the stability function after Businger et al. (1971), JAS, 28(2), 181–189

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: z,L
    REAL, INTENT(out)                                    :: stabfunc_busingerv

! Local
    REAL                                                 :: zL, X

!!!!!!! Variables
! z: height [m]
! L: Obukhov length [-]

    sfname = 'stabfunc_businger'

    IF (L /= 0.) THEN
      zL = z/L
    ELSE
      ! Neutral
      zL = 0.
    END IF

    IF (zL > 0.) THEN
    ! Stable case
      stabfunc_busingerv = 4.7*z/L
    ELSE IF (zL < 0.) THEN
    ! unstable
      X = (1. - 15.*z/L)**(0.25)
      !stabfunc_busingerv = -2.*LOG((1.+X)/2.)-LOG((1.+X**2)/2.)+2.*ATAN(X)-piconst/2.
      stabfunc_busingerv = LOG( ((1.+X**2)/2.)*((1.+X)/2.)**2)-2.*ATAN(X)+piconst/2.
    ELSE
      stabfunc_busingerv = 0.
    END IF

    RETURN

!  END FUNCTION stabfunc_businger
  END SUBROUTINE stabfunc_businger

  SUBROUTINE Cdrag_0(ust,uas,vas,Cd)
! Fuction to compute a first order generic approximation of the drag coefficient as
!   CD = (ust/wss)**2
!  after, Garratt, J.R., 1992.: The Atmospheric Boundary Layer. Cambridge Univ. Press, 
!    Cambridge, U.K., 316 pp
! Ackonwledgement: M. A. Jimenez, UIB
!
    IMPLICIT NONE

    REAL, INTENT(in)                                     :: ust, uas, vas
    REAL, INTENT(out)                                    :: Cd

!!!!!!! Variables
! ust: u* in similarity theory [ms-1]
! uas, vas: x/y-components of wind at 10 m
! Cd: Drag coefficient [-]

    sfname = 'Cdrag_0'
    IF (uas**2 + vas**2 /= 0.) THEN 
      Cd = ust**2/(uas**2+vas**2)
    ELSE
      Cd = 0.
    END IF

  END SUBROUTINE Cdrag_0

  SUBROUTINE var_potevap_bulk(rho1, cd, uas, vas, ts, ps, qv1, potevap)
! Subroutine to compute the generic potential evapotranspiration following simple bulk formulation
!  Manabe, S., (1969): Climate and the ocean circulation, 1. the atmospheric circulation and the hydrology of 
!    the earth's surface, Mon. Weather Rev., 97, 739-774
!      potevap = dt*rho1*qc*(q2sat-qv1)

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: rho1, cd, uas, vas, ts, ps, qv1
    REAL, INTENT(out)                                    :: potevap

! Local
    REAL                                                 :: qsol_sat, qc

!!!!!!! Variables
! rho1: atsmophere density at the first layer [kgm-3]
! cd: drag coefficient [-]
! uas, vas: x/y-components of 10-m wind [ms-1]
! ts: surface temperature [K]
! ps: surface pressure [Pa]
! qv1: 1st layer atmospheric mixing ratio [kgkg-1]
! potevap: potential evapo transpiration [kgm-2s-1]
  sfname = 'var_potevap_bulk'

  ! qsol_sat: Saturated air at ts
  qsol_sat = SaturationMixingRatio(ts, ps)

  ! qc: surface drag coefficient
  qc = SQRT(uas**2 + vas**2)*cd

  potevap = MAX(zeroRK, rho1*qc*(qsol_sat - qv1))

  END SUBROUTINE var_potevap_bulk

  SUBROUTINE var_potevap_Milly92(rho1, cd, uas, vas, tsk, ta1, ps, qv1, sfcevap, emiss, potevap)
! Subroutine to compute the potential bulk evapotranspiration with Milly 1992 correction
!   Milly, P. C. D. (1992): Potential evaporation and soil moisture in general circulation models, 
!     J. Climate, 5, 209–226
!      potevap = potevap_bulk*[1./(1.+correction_Milly)]
!        ETp(Ts) = potevap_bulk(Ts) = (dens/ra)*[qs(Ts) − qa]
!        ETp(Tv) = (dens/ra)*[qs(Tv) − qa]
!        correction_Milly = [ETp(Ts) - ETp(Tv)]/ETp(Tw)
!      see more details in:
!        A. Barella-Ortiz et al., (2013), Hydrol. Earth Syst. Sci., 17, 4625-4639
!
    IMPLICIT NONE

    REAL, INTENT(in)                                     :: rho1, cd, uas, vas, tsk, ta1, ps, qv1,    &
      sfcevap, emiss
    REAL, INTENT(out)                                    :: potevap

! Local
    REAL                                                 :: qsol_sat, qc
    REAL                                                 :: ta_05, ta05
    REAL                                                 :: potevapo_bulk, beta, derivT_qsol_sat
    REAL                                                 :: correction_Milly, corr_Milly1,            &
      corr_Milly2, corr_Milly

!!!!!!! Variables
! rho1: atsmophere density at the first layer [kgm-3]
! cd: drag coefficient [-]
! uas, vas: x/y-components of 10-m wind [ms-1]
! tsk: surface skin temperature
! ta1: 1st level air-temperature [K]
! ps: surface pressure [Pa]
! qv1: 1st layer atmospheric mixing ratio [kgkg-1]
! sfcevap: surface eavporation flux [Wm-2s-1]
! emiss: emissivity [1]
! potevap: potential evapo transpiration [kgm-2s-1]
  sfname = 'var_potevap_Milly92'

  ! qsol_sat: Saturated air by tsk
  qsol_sat = SaturationMixingRatio(tsk, ps)

  ! qc: surface drag coefficient
  qc = SQRT(uas**2 + vas**2)*cd

  ! Bulk potential evapotranspiration
  potevapo_bulk = rho1*qc*(qsol_sat-qv1)

  ! Moisture availability function (in a similar way is done in ORCHIDEE)
  !   beta = sfcevap / potevapo_bulk
  beta = sfcevap / potevapo_bulk

  ! derivT_qsol_sat: Derivative of Saturated air by ta1
  ! Using numerical 1st order approximation
  ! derivT_qsol_sat(T) = [qsol_sat(T+h) - qsol_sat(T-h)]/(2h)
  ta_05 = ta1-halfRK
  ta05 = ta1+halfRK
  derivT_qsol_sat = (SaturationMixingRatio(ta05, ps) - SaturationMixingRatio(ta_05, ps)) / (2.*halfRK)

  ! Milly's correction
  corr_Milly1 = XLV*rho1*qc*derivT_qsol_sat*(1.-beta)
  corr_Milly2 = 4.*emiss*STBOLT*ta1**3 + rho1*Cp*qc + XLV*rho1*qc*derivT_qsol_sat*beta
  correction_Milly = corr_Milly1 / corr_Milly2
  corr_Milly = 1./(1. + correction_Milly)

  potevap = MAX(zeroRK, potevapo_bulk*corr_Milly)

  END SUBROUTINE var_potevap_Milly92

  SUBROUTINE var_potevap_bulkgen(rho1, ust, uas, vas, ts, ps, qv1, potevap)
! Subroutine to compute the generic potential evapotranspiration following simple bulk formulation
!  Manabe, S., (1969): Climate and the ocean circulation, 1. the atmospheric circulation and the hydrology of 
!    the earth's surface, Mon. Weather Rev., 97, 739-774
!      potevap = dt*rho1*qc*(q2sat-qv1)

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: rho1, ust, uas, vas, ts, ps, qv1
    REAL, INTENT(out)                                    :: potevap

! Local
    REAL                                                 :: qsol_sat, Cd, qc

!!!!!!! Variables
! rho1: atsmophere density at the first layer [kgm-3]
! ust: u* in similarity theory [ms-1]
! uas, vas: x/y-components of 10-m wind [ms-1]
! ts: surface temperature [K]
! ps: surface pressure [Pa]
! qv1: 1st layer atmospheric mixing ratio [kgkg-1]
! potevap: potential evapo transpiration [kgm-2s-1]
  sfname = 'var_potevap_bulkgen'

  ! qsol_sat: Saturated air at ts
  qsol_sat = SaturationMixingRatio(ts, ps)

  ! Cd: drag coeffiecient
  CALL Cdrag_0(ust, uas, vas, Cd)

  ! qc: surface drag coefficient
  qc = SQRT(uas**2 + vas**2)*Cd

  potevap = MAX(zeroRK, rho1*qc*(qsol_sat - qv1))

  END SUBROUTINE var_potevap_bulkgen

  SUBROUTINE var_potevap_Milly92gen(rho1, ust, uas, vas, tsk, ta1, ps, qv1, sfcevap, emiss, potevap)
! Subroutine to compute the genric potential bulk evapotranspiration with Milly 1992 correction
!   Milly, P. C. D. (1992): Potential evaporation and soil moisture in general circulation models, 
!     J. Climate, 5, 209–226
!      potevap = potevap_bulk*[1./(1.+correction_Milly)]
!        ETp(Ts) = potevap_bulk(Ts) = (dens/ra)*[qs(Ts) − qa]
!        ETp(Tv) = (dens/ra)*[qs(Tv) − qa]
!        correction_Milly = [ETp(Ts) - ETp(Tv)]/ETp(Tw)
!      see more details in:
!        A. Barella-Ortiz et al., (2013), Hydrol. Earth Syst. Sci., 17, 4625-4639
!
    IMPLICIT NONE

    REAL, INTENT(in)                                     :: rho1, ust, uas, vas, tsk, ta1, ps, qv1,   &
      sfcevap, emiss
    REAL, INTENT(out)                                    :: potevap

! Local
    REAL                                                 :: qsol_sat, Cd, qc
    REAL                                                 :: ta_05, ta05
    REAL                                                 :: potevapo_bulk, beta, derivT_qsol_sat
    REAL                                                 :: correction_Milly, corr_Milly1,            &
      corr_Milly2, corr_Milly

!!!!!!! Variables
! rho1: atsmophere density at the first layer [kgm-3]
! ust: u* in similarity theory [ms-1]
! uas, vas: x/y-components of 10-m wind [ms-1]
! tsk: surface skin temperature
! ta1: 1st level air-temperature [K]
! ps: surface pressure [Pa]
! qv1: 1st layer atmospheric mixing ratio [kgkg-1]
! sfcevap: surface eavporation flux [Wm-2s-1]
! emiss: emissivity [1]
! potevap: potential evapo transpiration [kgm-2s-1]
  sfname = 'var_potevap_Milly92gen'

  ! qsol_sat: Saturated air by tsk
  qsol_sat = SaturationMixingRatio(tsk, ps)

  ! Cd: drag coeffiecient
  CALL Cdrag_0(ust, uas, vas, Cd)

  ! qc: surface drag coefficient
  qc = SQRT(uas**2 + vas**2)*Cd

  ! Bulk potential evapotranspiration
  potevapo_bulk = rho1*qc*(qsol_sat-qv1)

  ! Moisture availability function (in a similar way is done in ORCHIDEE)
  !   beta = sfcevap / potevapo_bulk
  beta = sfcevap / potevapo_bulk

  ! derivT_qsol_sat: Derivative of Saturated air by ta1
  ! Using numerical 1st order approximation
  ! derivT_qsol_sat(T) = [qsol_sat(T+h) - qsol_sat(T-h)]/(2h)
  ta_05 = ta1-halfRK
  ta05 = ta1+halfRK
  derivT_qsol_sat = (SaturationMixingRatio(ta05, ps) - SaturationMixingRatio(ta_05, ps)) / (2.*halfRK)

  ! Milly's correction
  corr_Milly1 = XLV*rho1*qc*derivT_qsol_sat*(1.-beta)
  corr_Milly2 = 4.*emiss*STBOLT*ta1**3 + rho1*Cp*qc + XLV*rho1*qc*derivT_qsol_sat*beta
  correction_Milly = corr_Milly1 / corr_Milly2
  corr_Milly = 1./(1. + correction_Milly)

  potevap = MAX(zeroRK, potevapo_bulk*corr_Milly)

  END SUBROUTINE var_potevap_Milly92gen

  SUBROUTINE var_fog_K84(qc, qi, fog, vis)
  ! Computation of fog (vis < 1km) only computed where qcloud, qice /= 0.
  ! And visibility following Kunkel, B. A., (1984): Parameterization of droplet terminal velocity and 
  !   extinction coefficient in fog models. J. Climate Appl. Meteor., 23, 34–41.

  IMPLICIT NONE

  REAL, INTENT(in)                                       :: qc, qi
  INTEGER, INTENT(out)                                   :: fog
  REAL, INTENT(out)                                      :: vis

! Local
  REAL                                                   :: visc, visi

!!!!!!! Variables
! qc: cloud mixing ratio [kgkg-1]
! qi, ice mixing ratio [kgkg-1]
! fog: presence of fog (1: yes, 0: no)
! vis: visibility within fog [km]

  sfname = 'var_fog_K84'
  
  IF (qi > nullv .OR. qc > nullv) THEN
    visc = 100000.*oneRK
    visi = 100000.*oneRK
    ! From: Gultepe, 2006, JAM, 45, 1469-1480
    IF (qc > nullv) visc = 0.027*(qc*1000.)**(-0.88)
    IF (qi > nullv) visi = 0.024*(qi*1000.)**(-1.0)
    ! Getting the lowest visibility
    vis = MINVAL((/visc, visi/))
    IF (vis <= oneRK) THEN
      fog = 1
    ELSE
      fog = 0
      vis = -oneRK
    END IF
  ELSE
    fog = 0
    vis = -oneRK
  END IF

  END SUBROUTINE var_fog_K84

  SUBROUTINE var_fog_RUC(rhv, fog, vis)
  ! Computation of fog (vis < 1km) only computed where qcloud, qice /= 0.
  ! And visibility following RUC method Smirnova, T. G., S. G. Benjamin, and J. M. Brown, 2000: Case 
  !   study verification of RUC/MAPS fog and visibility forecasts. Preprints, 9 th Conference on 
  !   Aviation, Range, and Aerospace Meteorlogy, AMS, Orlando, FL, Sep. 2000. Paper#2.3, 6 pp.

  IMPLICIT NONE

  REAL, INTENT(in)                                       :: rhv
  INTEGER, INTENT(out)                                   :: fog
  REAL, INTENT(out)                                      :: vis

! Local
  REAL                                                   :: rh

!!!!!!! Variables
! rh: relative humidty [1]
! vis: visibility within fog [km]

  sfname = 'var_fog_RUC'

  ! Avoiding supersaturation 
  rh = MINVAL((/1.,rhv/))

  IF (rh > 0.3) THEN
    ! From: Gultepe, I., and G. Isaac, 2006: Visbility versus precipitation rate and relative 
    !   humidity. Preprints, 12th Cloud Physics Conf, Madison, WI, Amer. Meteor. Soc., P2.55. 
    !   [Available  online  at  http://ams.confex.com/ams/Madison2006/techprogram/paper_l13177.htm]
    vis = 60.*EXP(-2.5*(rh*100.-15.)/80.)
    IF (vis <= oneRK) THEN
      fog = 1
    ELSE
      fog = 0
      vis = -oneRK
    END IF
  ELSE
    fog = 0
    vis = -oneRK
  END IF

  END SUBROUTINE var_fog_RUC

  SUBROUTINE var_fog_FRAML50(rhv, fog, vis)
  ! Computation of fog (vis < 1km)
  ! And visibility following Gultepe, I. and J.A. Milbrandt, 2010: Probabilistic Parameterizations 
  !   of Visibility Using Observations of Rain Precipitation Rate, Relative Humidity, and Visibility. 
  !   J. Appl. Meteor. Climatol., 49, 36-46, https://doi.org/10.1175/2009JAMC1927.1
  ! Interest is focused on a 'general' fog/visibilty approach, thus the fit at 50 % of probability
  !   is chosen
  ! Effects from precipitation are not considered

  IMPLICIT NONE

  REAL, INTENT(in)                                       :: rhv
  INTEGER, INTENT(out)                                   :: fog
  REAL, INTENT(out)                                      :: vis

! Local
  REAL                                                   :: rh

!!!!!!! Variables
! rhv: relative humidty [1]
! fog: presence of fog (1: yes, 0: no)
! vis: visibility within fog [km]

  sfname = 'var_fog_FRAML50'

  ! Avoiding supersaturation 
  rh = MINVAL((/1.,rhv/))

  IF (rh > 0.3) THEN
    vis = -5.19*10.**(-10)*(rh*100.)**5.44+40.10
    ! Fog definition (vis <= 1. km)
    IF (vis <= oneRK) THEN
      fog = 1
    ELSE
      vis = -oneRK
      fog = 0
    END IF
  ELSE
    vis = -oneRK
    fog = 0
  END IF

  END SUBROUTINE var_fog_FRAML50

  SUBROUTINE var_tds(hurs, tas, tds)
! Subroutine to compute dew point temperature at 2m using August-Roche-Magnus approximation [k]

    IMPLICIT NONE

    REAL, INTENT(in)                                     :: hurs, tas
    REAL, INTENT(out)                                    :: tds

! Local
    REAL                                                 :: gammatarh

!!!!!!! Variables
! hurs: 2m relative humidity [1]
! tds: dew point temperature [k]

    sfname = 'var_tds'

    gammatarh = LOG(hurs) + ARM2*(tas-SVPT0)/((tas-SVPT0)+ARM3)
    tds = ARM3*gammatarh/(ARM2-gammatarh) + SVPT0

    RETURN

  END SUBROUTINE var_tds

END MODULE module_diagvar_cordex
